#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <errno.h>

#include "xmdefs.h"
#include "xmerror.h"

#define  Max(x, y) ((x) > (y))? (x):(y)
#define  SOCK_TABLE_SIZE 16

static int  xmOpenXMSocket(XMDescriptor *);
static int  xmNewClientConnection(XMSession *, XMDescriptor *, char **);
static void xmDeleteSession(XMSession **, short, int *, SocketTable *, int);
static int  xmNewSession(XMSession **, int *);
static void xmInsertSocketDesc(SocketTable **, int *, int, short, 
                               unsigned char);
static void xmDeleteSocketDesc(SocketTable *, int, int);
static void xmAddToTable(SocketTable **, int *, XMSession *, short);
static void xmDeleteFromTable(SocketTable *, int, XMSession *);
static Boolean xmDataAvailable(register XMSession *, int, unsigned char, 
                                 int *);
static void printXMS(XMSession *);

/* Wait for a client to connect to and then handle a session with a list of 
   servers */
void 
xmMultiplexer(char **xservers_list)
{
  XMSession *sessions[MAX_SESSIONS];  /* Table of sessions.  A session id 
                                         is the index */
  int num_sessions;     /* Number of sessions */
  SocketTable *sockTable;  /* Table of sockets currently in use with 
                              their mapping to session ids */
  int sockTableSize;    /* Number of sockets currently in use */
  XMDescriptor xm_desc; /* Port number, socket on which xm listens for 
                           clients */
  short session_id;  /* Temporary */

  /* There are no sessions yet.  No sockets have been created yet.
     Initialize state  */
  num_sessions = 0;
  for(session_id = 0;session_id < MAX_SESSIONS;session_id++)
    sessions[session_id] = NULL;

  sockTable = NULL;
  sockTableSize = 0;

  printf("xm listening on port %d for connections\n", 
                                              xmOpenXMSocket(&xm_desc));
  
  /* Maximum of five X clients queuing up */
  if (listen(xm_desc.xm_socket, 5) < 0) {
    xmerror = LISTENERROR;
    xmErrorQuitSYS("xmMultiplexer() - listen error\n");
  }
  
  /* Allocate buffer space */
  buffer = (char *)malloc(BUFSIZE);
  assert(buffer != NULL);

  do {
    fd_set readfrom;      /* Set of end points that have data ready */
    int enable;
    int socket;  /* Temporary */
    Boolean exception; 
    int sender_id; /* For identifying the sender */

    FD_ZERO(&readfrom); /* We do not have anyone talking yet */

    for(session_id = 0;session_id < MAX_SESSIONS;session_id++)
      if (sessions[session_id] != NULL) {
        if (sessions[session_id]->num_await == 0)
          enable = XSERVER | XCLIENT;
        else 
          enable = XSERVER;
        xmFDInit(sessions[session_id], &readfrom, enable); 
      }

    /* Clients may connect asynchronously too */
    FD_SET(xm_desc.xm_socket, &readfrom);

    /* Wait until at least one of the open sockets has data */
    if (select(Max(sockTableSize, xm_desc.xm_socket+1), &readfrom, 
               (fd_set *) NULL, (fd_set *) NULL, 
               (struct timeval *) NULL) < 0) {
      xmerror = SELECTERROR;
      xmErrorQuitSYS("xmMultiplexer() select error\n");
    }

    /* Is there a new x client asking for a connection? */
    if (FD_ISSET(xm_desc.xm_socket, &readfrom)) {
      FD_CLR(xm_desc.xm_socket, &readfrom);
      if ((session_id = xmNewSession(sessions, &num_sessions)) != ERROR) {
        xmInitialize(sessions[session_id]);
        sessions[session_id]->session_id = session_id;
        if (xmNewClientConnection(sessions[session_id], &xm_desc, 
            xservers_list) != ERROR)
          xmAddToTable(&sockTable, &sockTableSize, sessions[session_id], 
                     session_id);
        else {
          xmDeleteSession(sessions, session_id, &num_sessions,
                                   NULL, 0);
          continue;
        }
      }
      else
        xmErrorNoQuit("xmMultiplexer - Cannot start new session\n");
    }
    
    /* Check for data on other ports */
    for(socket = 0;socket < sockTableSize;socket++)
      if (FD_ISSET(socket, &readfrom)) {
        session_id = sockTable[socket].session_id;
        assert(session_id != ERROR);

        exception = xmDataAvailable(sessions[session_id], socket,
                                           sockTable[socket].type, 
                                           &sender_id);
        if (exception) {
          /* There is a unrecoverable exception only if either the client
             to the master X server was involved. */
          switch(sockTable[socket].type) {
            case XSERVER:
                 if (sender_id == MASTER_XSERVER) 
                   xmDeleteSession(sessions, session_id, &num_sessions,
                                   sockTable, sockTableSize);
                 else {
                   xmDeleteSocketDesc(sockTable, sockTableSize, 
                   sessions[session_id]->xservers[sender_id]->socketd);
                   xmErrorNoQuit("Connection to server on %s lost\n", 
                           sessions[session_id]->xservers[sender_id]
                                                              ->xserver_name);
                   xmCloseServerConnection(sessions[session_id], 
                                           sender_id);
                 }
                 break;
  
            case XCLIENT:
                   xmDeleteSession(sessions, session_id, &num_sessions, 
                     		sockTable, sockTableSize);
                   break;
          }

          /* Once a session has been terminated, no longer read data from
             sockets belonging to the terminated session */
          if (sessions[session_id] == NULL)
            FD_ZERO(&readfrom);
        }
        else if (sockTable[socket].type == XSERVER &&
                 sender_id == MASTER_XSERVER && 
                 sessions[session_id]->ResizeReqBuf != NULL) {

					if (sessions[session_id]->ReSize == NOT_FIRST_TIME) {
						xmMakeConfWinPkt(sessions[session_id]);
						xmMultiCast(sessions[session_id], SLAVE_SERVERS_ONLY);
						sessions[session_id]->numFakeReqs++;
					}
					else /* The first time */
						sessions[session_id]->ReSize = NOT_FIRST_TIME;

					free(sessions[session_id]->ResizeReqBuf);
					sessions[session_id]->ResizeReqBuf = NULL;
				}
      }  
  } while(1);
}

/* Opens a socket on which xm listens for client connections.  If socket can
   not be opened, quit. */
static int
xmOpenXMSocket(XMDescriptor *xm_desc)
{
  /* Open a socket to listen to clients that want to connect */
  if ((xm_desc->xm_socket = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    xmerror = SOCKETOPENERROR;
    xmErrorQuitSYS("xmMultiplexer() Cannot open socket to connect to server\n");
  }
  
  /* Fill in addresstype, inet address, port number */
  memset((char *) &xm_desc->xm_address, 0, sizeof(struct sockaddr_in));
  xm_desc->xm_address.sin_family = AF_INET;
  xm_desc->xm_address.sin_addr.s_addr = htonl(INADDR_ANY);

  for(xm_desc->xm_port = FIRST_PORT;xm_desc->xm_port < LAST_PORT;
      xm_desc->xm_port++) {
    xm_desc->xm_address.sin_port = htons(xm_desc->xm_port);

    /* bind the X multiplexer address and the port number to the socket */
    if (bind(xm_desc->xm_socket, (struct sockaddr *) &xm_desc->xm_address, 
                 sizeof(struct sockaddr_in)) == 0)
      break;   /* Found a port to listen on */
  }
  
  if (xm_desc->xm_port == LAST_PORT) {
    xmerror = BINDERROR;
    xmErrorQuitSYS("xmMultiplexer() Cannot bind address\n");
  }

  return xm_desc->xm_port;
}
  
static int
xmNewClientConnection(XMSession *session, XMDescriptor *xm_desc, 
                      char **xservers_list)
{
    int retVal;

    /* Wait to accept connections */
    xmAcceptClientConnection(session, xm_desc);

    /* Connect to the X servers specified on the command line.
			 xmStartServerSession returns number of servers connected to */
    if (xmStartServerSession(xservers_list, session) == 0)
			return ERROR;

    /* Now that a client has connected, wait for all X servers to respond
       to the setup connection request.  */
    retVal = xmCompleteConnection(session); 
		printf("Returned from xmCompleteConnection %d\n", retVal);
		return retVal;
}

static int
xmNewSession(XMSession **sessions, int *num_sessions)
{
  int session_id;

  if (*num_sessions == MAX_SESSIONS) {
    xmErrorNoQuit("Cannot accept any more connections\n");
    return ERROR;
  }

  for(session_id = 0;session_id < MAX_SESSIONS;session_id++)
    if (sessions[session_id] == NULL) {
      sessions[session_id] = (XMSession *)malloc(sizeof(XMSession));
      assert(sessions[session_id] != NULL);
      memset(sessions[session_id], 0, sizeof(XMSession));
      break;
    }

  (*num_sessions)++;
  return session_id;
}
  
static void
xmDeleteSession(XMSession **sessions, short session_id, int *num_sessions,
                SocketTable *sockTable, int sockTableSize)
{
  assert(*num_sessions != 0);
  assert(sessions[session_id] != NULL);

	if (sockTable != NULL)
	  xmDeleteFromTable(sockTable, sockTableSize, sessions[session_id]);

  xmCloseClientConnection(sessions[session_id]);
  xmCloseServerConnection(sessions[session_id], ALL_SOCKETS);

  free(sessions[session_id]);
  sessions[session_id] = NULL;
  (*num_sessions)--;
}

static void
xmAddToTable(SocketTable **sockTable, int *sockTableSize, XMSession *session, 
           short session_id)
{
  int serv_id;

  xmInsertSocketDesc(sockTable, sockTableSize, session->xclient->socketd, 
                     session_id, XCLIENT);
  for(serv_id = 0;serv_id < MAX_CONNECTIONS;serv_id++)
    if (session->xservers[serv_id] != NULL)
      xmInsertSocketDesc(sockTable, sockTableSize, 
                         session->xservers[serv_id]->socketd, session_id, 
                         XSERVER);
}

static void 
xmInsertSocketDesc(SocketTable **sockTable, int *sockTableSize, int socket,
                   short session_id, unsigned char serv_or_cli)
{
  while (socket >= *sockTableSize) {
    *sockTable = (SocketTable *)realloc(*sockTable, 
                        (*sockTableSize+SOCK_TABLE_SIZE)*sizeof(SocketTable));
    assert(*sockTable != NULL);
    memset(*sockTable+*sockTableSize, ERROR, 
           sizeof(SocketTable)*SOCK_TABLE_SIZE);
    *sockTableSize += SOCK_TABLE_SIZE;
  }
  (*sockTable)[socket].session_id = session_id;
  (*sockTable)[socket].type = serv_or_cli;
}

static void 
xmDeleteFromTable(SocketTable *sockTable, int sockTableSize, 
                  XMSession *session)
{
  int serv_id;

  xmDeleteSocketDesc(sockTable, sockTableSize, session->xclient->socketd);

  for(serv_id = 0;serv_id < MAX_CONNECTIONS;serv_id++)
    if (session->xservers[serv_id] != NULL)
      xmDeleteSocketDesc(sockTable, sockTableSize, 
                       session->xservers[serv_id]->socketd);
    
}

static void 
xmDeleteSocketDesc(SocketTable *sockTable, int sockTableSize, int socket)
{
	assert(sockTable != NULL);
  assert(socket < sockTableSize);
  assert(sockTable[socket].session_id != ERROR);

  sockTable[socket].session_id = ERROR;
  sockTable[socket].type = 0;
}

/* We have data on one of the open sockets.  Check for the open socket and
   shunt out the data on the other end.  */
static Boolean
xmDataAvailable(register XMSession *session, int socket, 
                unsigned char serv_or_cli, int *sender_id)
{
  int serv_id; /* For iteration */
  Boolean exception = FALSE;

  assert(session != NULL);

  if (serv_or_cli & XSERVER) {

    /* Which server has data? */
    for(serv_id = 0; serv_id < MAX_CONNECTIONS; serv_id++) 
      /* Is there an X server in this slot? */
      if (session->xservers[serv_id] != NULL 
          && session->xservers[serv_id]->socketd == socket) 
        break; 

      assert(serv_id < MAX_CONNECTIONS);

      *sender_id = serv_id;
      if (xmReadServer(session, serv_id) == 0) {
        exception = TRUE;
        xmerror = SERVERSOCKETREADERROR;
      }
  } 
  else {
    if (xmReadClient(session) == 0) {
      exception = TRUE;
      xmerror = CLIENTSOCKETREADERROR; 
    }
  }

  return exception;
}
    
/* Enable "read" on client and server side */
int
xmFDInit(register XMSession *session, fd_set *readfrom, int side)
{
  int serv_id;
	int maxSocketDesc = -1;

  assert(session != NULL);

  /* Do we need to enable the client side? */
  if (side & XCLIENT) {
    FD_SET(session->xclient->socketd, readfrom);
    maxSocketDesc = session->xclient->socketd;
  }

  /* Do we need to enable the server side? */
  if (side & XSERVER)
    for(serv_id = 0; serv_id < MAX_CONNECTIONS; serv_id++)
      if (session->xservers[serv_id] != NULL) {
        FD_SET(session->xservers[serv_id]->socketd, readfrom);
				if (maxSocketDesc < session->xservers[serv_id]->socketd)
          maxSocketDesc = session->xservers[serv_id]->socketd;
      }

  return maxSocketDesc;
}


/* Initializes all variables before a session starts */
void
xmInitialize(XMSession *session)
{
  int serv_id;
    
  session->num_xservers = 0;
  session->num_await =  0;
  session->await_table_size =  0;
  session->await = NULL;

  for(serv_id = 0;serv_id < MAX_CONNECTIONS; serv_id++)
    session->xservers[serv_id] = NULL;

  session->xclient = NULL;
  session->buffer = buffer;
  session->max_buffer_len = BUFSIZE;
  session->curr_buffer_len = 0;
	session->seq_num_last_processed = 0;
	session->ReSize = FIRST_TIME;
	session->ResizeReqBuf = NULL;
	session->numFakeReqs = 0;
}

static void
printXMS(XMSession *session) 
{
  int i;

  printf("Num X Servers = %d\n", session->num_xservers);

  for(i = 0;i < MAX_CONNECTIONS;i++)
    if (session->xservers[i] != NULL)
      printf("Server = %s Socket = %d\n", session->xservers[i]->xserver_name,
                                          session->xservers[i]->socketd);
  printf("Client = %s Socket = %d\n", session->xclient->xclient_name,
                                      session->xclient->socketd);
}
