#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <errno.h>
#include <assert.h>

#include <X11/X.h> /* For typedef of Atom */
#include "xmdefs.h"
#include "xmerror.h"


int
xmStartServerSession(char **namelist, XMSession *serv_session)
{
  char name[MAXHOSTNAMELENGTH];
  int port;
  struct hostent *xserver_info;
  XServer *current = NULL;  /* For easier access to "current" entry */

  /* Initialize server count */
  serv_session->num_xservers = 0;

  while(*namelist != NULL) {

    /* "current" points to entry being filled in */
    if (current == NULL) {
      current = (XServer *)malloc(sizeof(XServer));
      assert(current != NULL);
    }

    /* Check if we have a valid server:port pair */
    if (xmGetServerPort(*namelist++, name, &port)) {

      /* Get information for each server listed.  Make an entry only if we have 
         a valid server */
      if ((xserver_info = xmGetHostInfo(name)) != NULL) {

        /* We also need to check if we can connect to the X server before we 
           make an entry.  We do not want entries of hosts with no X server 
           running or that do not accept connections */
        if (!xmConnectToServer(current, XPORTS_BASE+port, xserver_info)) {
          xmerror = SERVERCONNECTERROR;
          xmErrorNoQuit("xmStartServerSession() cannot connect to 
                         server %s Port = %d\n", name, XPORTS_BASE+port);
          continue;
        }

        /* Connection successful.  Fill in host name, port number connected to 
           and increment server count */
        current->xserver_name = (char *)malloc(strlen(xserver_info->h_name) + 1);
        assert(current->xserver_name != NULL);
        strcpy(current->xserver_name, xserver_info->h_name);
        current->port = XPORTS_BASE+port;
        current->serv_info = NULL;
        current->atoms = (Atom *)malloc(ATOMS_TABLE_SIZE*sizeof(Atom));
        assert(current->atoms != NULL);
        current->size_atoms = ATOMS_TABLE_SIZE;
        serv_session->xservers[serv_session->num_xservers] = current;
        serv_session->num_xservers++;

        xmDEBUG("Connected to %s Port = %d\n", current->xserver_name, 
                                               current->port);

        current = NULL;
      }
      else {
        /* There was a problem with getting information on the host */
        xmerror = HOSTINFOERROR;
				xmErrorNoQuit("%s - ", *(namelist-1));
        perror("");
      }
    }
    else {
      xmerror = PARSEERROR;
      xmErrorNoQuit("Cannot parse %s\n", *(namelist-1));
    }
  } /* for */

  if (current != NULL)
    free(current);

  return serv_session->num_xservers;
}


int
xmAcceptClientConnection(XMSession *session, XMDescriptor *xm_desc)
{
  int client_addr_len;
  struct hostent *client_info;
  char *dot_address;
  XClient *client;

  client = (XClient *)malloc(sizeof(XClient));
  assert(client != NULL);

  /* Length of the client's address */
  client_addr_len = sizeof(struct sockaddr);

  /* Wait to accept connections */
  if ((client->socketd = accept(xm_desc->xm_socket, 
       (struct sockaddr *) &client->xclient_address, &client_addr_len)) < 0) {
    xmerror = CLIENTACCEPTERROR;
    xmErrorNoQuitSYS("xmWaitForClient() - Cannot accept connection\n");
    return ERROR;
  }

  /* Convert to dot address.  We will need it later */
  dot_address = inet_ntoa(client->xclient_address.sin_addr);

  /* X client can connect to this server.  Get more info on the X client */
  if ((client_info = xmGetHostInfo(dot_address)) == NULL) {
    xmerror = HOSTINFOERROR;
    xmErrorNoQuitSYS("xmWaitForClient()");
  }
  else {
    client->xclient_name = (char *)malloc(strlen(client_info->h_name)+1);
    assert(client->xclient_name != NULL);
    strcpy(client->xclient_name, client_info->h_name);
  }

  xmDEBUG("Client from %s connected\n", client->xclient_name);

  client->sequence_num = 0;
  session->xclient = client;

  return 0;
}


/* Close connection on client side. */
void 
xmCloseClientConnection(XMSession *session)
{
  assert(session->xclient != NULL);

  if (close(session->xclient->socketd) == -1) {
    xmErrorNoQuitSYS("");
    xmerror = SOCKETCLOSEERROR;
  }

  xmDEBUG("X client connection from %s shutdown\n",
                                             session->xclient->xclient_name);
  free(session->xclient->xclient_name);
  free(session->xclient);
  session->xclient = NULL;
}

/* Close connection on server side.  index will have index of connection to 
   close.  If index == ALL_SOCKETS, close all connections.  */
void 
xmCloseServerConnection(XMSession *session, int index)
{
  int i, j;  /* For iteration */
  int num_roots;

  for(i = 0;i < MAX_CONNECTIONS; i++) {
    /* Check if only one socket needs to be shutdown.  If all need to be shut 
       down, index will be ALL_SOCKETS */
    if (index != ALL_SOCKETS)
      i = index;
        
    if (session->xservers[i] != NULL) {
      if (close(session->xservers[i]->socketd) == -1) {
        xmerror = SOCKETCLOSEERROR;
        xmErrorNoQuitSYS("");
      }

      xmDEBUG("Connection to X server on %s shutdown\n",
            session->xservers[i]->xserver_name);

      session->num_xservers--;

      /* This test is needed as the multiplexer may be killed before a client 
         connects and the server parameters are filled */
      if (session->xservers[i]->serv_info != NULL) {
        num_roots = session->xservers[i]->serv_info->setup->numRoots;

        /* Free space allocated to store pointers to Depths and Window roots */
        for(j = 0; j < num_roots; j++) {
          ListxWindowRoot *tmp_wnd = 
                    &session->xservers[i]->serv_info->list_xwindow_root[j];
          free(tmp_wnd->list_xdepth);
        }
        free(session->xservers[i]->serv_info->list_xwindow_root);
        free(session->xservers[i]->serv_info->prefix);
        free(session->xservers[i]->serv_info);
      }
      free(session->xservers[i]->atoms); 
      free(session->xservers[i]->xserver_name); 
      free(session->xservers[i]); 
      session->xservers[i] = NULL;

      if (index != ALL_SOCKETS)
        return;
    }
  }
}

void 
xmCloseAll(XMSession *session)
{
  xmCloseClientConnection(session);
  xmCloseServerConnection(session, ALL_SOCKETS);
}
