package rbs.ui;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.lang.reflect.InvocationTargetException;
import java.lang.NoSuchMethodException;
import java.lang.IllegalAccessException;
import java.util.HashMap;
import java.util.ArrayList;

import rbs.SimulationState;

public class CommandLineInterface extends UserInterface{
	private static final String HELP_COMMAND = "help";
	private static final String QUIT_COMMAND = "quit";
	private String HELP_TEXT = "\nCommand Summary\n" +
		"==========================================\n" + 
		"help\tDisplays a list of commands\n" + 
		"quit\tQuits the simulation\n";
	
	private static BufferedReader input = new BufferedReader(
			new InputStreamReader(System.in));

	private ArrayList<String> commands;
	private HashMap<String, Method> commandMap;
	private HashMap<String, Object> objMap;
	private HashMap<String, String> promptMap;
	private String helpText;

	public CommandLineInterface(){
		commandMap = new HashMap<String, Method>();
		objMap = new HashMap<String, Object>();
		promptMap = new HashMap<String, String>();
		commands = new ArrayList<String>();
		helpText = HELP_TEXT;
	}

	public void start(){
		String line;

		while(!SimulationState.getInstance().isFinished()){
			displayPrompt();
			line = readInput();
			handleInput(line);
		}

		System.out.println("Goodbye");
	}

	public void initInterface(){
		//create map between commands and functions
		addCommand(HELP_COMMAND, loadMethod(this.getClass(), HELP_COMMAND));
		objMap.put(HELP_COMMAND, this);
		addCommand(QUIT_COMMAND, loadMethod(this.getClass(), QUIT_COMMAND));
		objMap.put(QUIT_COMMAND, this);
	}

	private Method loadMethod(Class<?> owner, String method){
		Method theMethod = null;
		try{
			theMethod = owner.getMethod(method);
		} catch(NoSuchMethodException e){
			System.out.println("ERROR: Could not load method, " + method);
			SimulationState.getInstance().setFinished(true);
		}
		return theMethod;
	}

	private Method loadStringMethod(Class<?> owner, String method){
		Method theMethod = null;
		try{
			theMethod = owner.getMethod(method, String.class);
		} catch(NoSuchMethodException e){
			System.out.println("ERROR: Could not load method, " + method);
			SimulationState.getInstance().setFinished(true);
		}
		return theMethod;
	}

	protected void addCommand(String command, Method method){
		commandMap.put(command, method);
	}

	public String readInput(){
		try{
			return input.readLine();
		} catch(IOException e){
			//logger.severe("Could not read standard input");
			e.printStackTrace();
			SimulationState.getInstance().setFinished(true);
		}
		return "";
	}

	public void displayPrompt(){
		System.out.print("rbs> ");
	}

	public void handleInput(String input){
		//lookup input in map and call method if found
		Method method = this.commandMap.get(input);
		Object obj = this.objMap.get(input);
		if(method != null && obj != null){
			try{
				//need to check if method needs arguments if so 
				Type[] args = method.getGenericParameterTypes();
				if(args.length > 0){
					String arg = getInput(this.promptMap.get(input));
					method.invoke(obj, arg);
				}
				else{
					method.invoke(obj);
				}
			} catch(IllegalAccessException e){
				//logger.warning("Could not execute requested command");
			} catch(InvocationTargetException e){
				//logger.warning("Could not execute requested command");
			}
		}
		else{
			System.out.println("Unknown command");
		}
	}

	private String getInput(String prompt){
		System.out.print("\t" + prompt + "> ");
		return readInput();
	}

	public void help(){
		System.out.println(this.helpText);
	}

	public void quit(){
		SimulationState.getInstance().setFinished(true);
	}

	//should be able to refactor these with one parent interface
	public void addPlaybackCommands(Playback playback){
		//add commands
		for(int index = 0; index < Playback.COMMANDS.length; index++){
			objMap.put(Playback.COMMANDS[index], playback);
			addCommand(Playback.COMMANDS[index],
				loadMethod(playback.getClass(), Playback.COMMANDS[index]));
		}
		//add to help text
		this.helpText = this.helpText.concat(Playback.HELP_TEXT);
	}

	public void addSerializableReelCommands(SerializableReel reel){
		for(int index = 0; index < SerializableReel.COMMANDS.length; index++){
			objMap.put(SerializableReel.COMMANDS[index], reel);
			promptMap.put(SerializableReel.COMMANDS[index],
				SerializableReel.PROMPTS[index]);
			addCommand(SerializableReel.COMMANDS[index],
				loadStringMethod(reel.getClass(), 
					SerializableReel.COMMANDS[index]));
		}
		this.helpText = this.helpText.concat(SerializableReel.HELP_TEXT);
	}

	//command functions need to be implemented in other places
	//via interfaces
	/*public void impulse(){
		//get the id of the object experiencing the impluse
		System.out.print("\tObject ID> ");
		int oid = Integer.parseInt(this.readInput());
		System.out.print("\tTime > ");
		float time = Float.parseFloat(this.readInput());
		//x component of force
		System.out.print("\tForce X> ");
		float fx = Float.parseFloat(this.readInput());
		//y component of force
		System.out.print("\tForce Y> ");
		float fy = Float.parseFloat(this.readInput());
		//z component of force
		System.out.print("\tForce Z> ");
		float fz = Float.parseFloat(this.readInput());
		sim.newImpulseEvent(oid, time, fx, fy, fz);
	}

	public void pause(){
		SimulationState.getInstance().setPaused(true);
	}*/
}
