package rbs.sim;

import java.lang.Runnable;
import java.lang.Thread;
import java.lang.InterruptedException;
import java.lang.CloneNotSupportedException;

import java.util.logging.Logger;
import java.util.PriorityQueue;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.FileNotFoundException;
import java.io.IOException;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import rbs.SimulationState;
import rbs.render.Renderable;
import rbs.render.RenderableFactory;
import rbs.reel.FrameReel;
import rbs.reel.StillFrame;

public abstract class SimulationThread implements Runnable{
	public static Logger logger =
		Logger.getLogger(SimulationThread.class.getName());

	public enum COLLISION_RESP{NONE, HANDLED, OVERLAP}

	private Thread simulationThread;
	private FrameReel reel;
	private Object3D[] objects;
	private Object3D[] modifiedObjects;
	private boolean modified;
	private int numObjects;
	private PriorityQueue<ForceEvent> eventQueue;

	private float currentTime;
	private float endTime;
	private float timeDelta;

	private boolean pauseOnCollision;

	private Vector3f gravity;
	private boolean gravityForce;

	public SimulationThread(String fileName, boolean pauseOnCollision,
			Vector3f gravity, boolean isGravityForce){
		eventQueue = new PriorityQueue<ForceEvent>();
		simulationThread = new Thread(this);
		simulationThread.start();
		SimulationState.getInstance().configLocalLogger(logger);

		this.gravity = new Vector3f(gravity);
		this.gravityForce = isGravityForce;
		this.modified = false;

		logger.info("Loading Objects from " + fileName);
		loadObjectsFromFile(fileName);
		logger.info("Loading Objects from " + fileName + " complete");
		currentTime = 0.0f;
		logger.info("Current Sim Time: " + currentTime);
		this.pauseOnCollision = pauseOnCollision;
		logger.config("Pause On Collision = " + pauseOnCollision);
	}

	public synchronized Vector3f getGravity(){
		return this.gravity;
	}

	public synchronized boolean isGravityForce(){
		return this.gravityForce;
	}

	protected synchronized float getCurrentTime(){
		return this.currentTime;
	}

	protected synchronized void setCurrentTime(float time){
		this.timeDelta = time - this.currentTime;
		this.currentTime = time;
	}

	protected synchronized PriorityQueue<ForceEvent> getEventQueue(){
		return this.eventQueue;
	}

	/*public synchronized void newImpulseEvent(int objId, float time, 
		float forceX, float forceY, float forceZ){
		//verify the input if out of range log it and move on
		if(!this.isTimeInFuture(time)){
			logger.warning("Invalid Event Time: " + time);
			return;
		}

		if(!this.objectIndexCheck(objId)){
			logger.warning("Invalid Object Id: " + objId);
			return;
		}

		//create a new impulse force event
		ImpulseForceEvent event = new ImpulseForceEvent(objects[objId], time,
			forceX, forceY, forceZ);

		//put it on the priority queue
		this.eventQueue.add(event);
	}*/

	public synchronized int getNumObjects(){
		return this.numObjects;
	}

	protected synchronized void setEndTime(float endTime){
		this.endTime = endTime;
		logger.config("Simulation will end after " + this.endTime);
	}

	public synchronized float getEndTime(){
		return this.endTime;
	}

	public synchronized void setReel(FrameReel reel){
		this.reel = reel;
		logger.info("Simulation reel set to " + reel);
	}

	public synchronized FrameReel getCurrentReel(){
		return this.reel;
	}

	protected synchronized Object3D[] getObjects(){
		if(modified)
			return this.modifiedObjects;
		else
			return this.objects;
	}

	protected abstract void advanceTime();
	protected abstract void rollbackTime();
	protected abstract void resumeTime();
	protected abstract COLLISION_RESP collisionCheck();

	//called once for each time step
	private synchronized void update(){
		//check to see if an event occured at this time if so need to have
		//the event do its thing
		//TODO: fix event handling when we rollback we could lose events
		ForceEvent currentEvent = this.eventQueue.peek();
		while(currentEvent != null && 
			currentEvent.getTime() == this.currentTime){
			currentEvent.handleIt();
			//remove the event we just handled
			this.eventQueue.poll();
			currentEvent = this.eventQueue.peek();
		}

		//loop through each object update current location
		//distance traveled = velocity * time
		for(int object = 0; object < this.numObjects; object++){
			logger.fine("Updating object " + object);

			try{
				modifiedObjects[object] = (Object3D)objects[object].clone();
			} catch(CloneNotSupportedException e){}

			modifiedObjects[object].update(this.timeDelta);
		}
		this.modified = true;
	}

	private synchronized StillFrame generateFrame(boolean collision){
		StillFrame frame = new StillFrame(this.numObjects);
		for(int object = 0; object < this.numObjects; object++){
			logger.info("Cloning object " + object);
			//make a copy of the 3d object
			try{
				frame.addModel((Renderable)objects[object].getModel().clone());
			} catch(CloneNotSupportedException e){
				logger.warning("Could not clone object");
			}
			logger.info("Cloning complete");
		}
		logger.info("Frame should be renderered until: " + this.currentTime);
		frame.setEndTime(this.currentTime);
		if(this.currentTime == this.getEndTime() || 
			isOverTime(this.currentTime)){
			frame.setLast(true);
		}
		if(collision && this.pauseOnCollision){
			frame.setPause(true);
		}
		return frame;
	}

	//to save on not running this loop it could be moved into generateFrame
	//since by the time we get to generateFrame the changes should be verified
	private synchronized void commitChanges(){
		if(modified){
			for(int object = 0; object < this.numObjects; object++){
				try{
					this.objects[object] = 
						(Object3D)this.modifiedObjects[object].clone();
				} catch(CloneNotSupportedException e){}
			}
			modified = false;
		}
	}

	private synchronized void rollbackChanges(){
		if(modified){
			modified = false;
		}
	}

	public void run(){
		boolean collision = false;
		//first check simulation state object to see if we should shutdown
		while(!SimulationState.getInstance().isFinished()){
			//check to see if we have a reel and not paused
			if(this.getCurrentReel() != null &&
					!this.isOverTime(this.currentTime)){

				logger.info("BEGIN FRAME: " + this.currentTime);

				logger.info("Updating Objects");
				this.update();

				logger.info("Checking For Object Collisions");
				COLLISION_RESP resp = this.collisionCheck();
				if(resp == COLLISION_RESP.NONE){
					this.commitChanges();
					collision = false;
				}
				else if(resp == COLLISION_RESP.HANDLED){
					this.commitChanges();
					this.resumeTime();
					collision = true;
				}
				else if(resp == COLLISION_RESP.OVERLAP){
					this.rollbackChanges();
					this.rollbackTime();
					collision = false;
				}

				this.advanceTime();
				this.getCurrentReel().pushFrame(this.generateFrame(collision));
				logger.info("Frame added to reel " + this.reel);
				logger.info("END FRAME");
			}
			else{
				try{
					Thread.sleep(100);
				} catch(InterruptedException e){
					break;
				}
			}
		}
		logger.info("Simulation Complete");
	}

	private synchronized boolean isTimeInPast(float time){
		if(time > 0.0f && time < this.currentTime)
			return true;
		else
			return false;
	}

	private synchronized boolean isTimeInFuture(float time){
		if(time > this.currentTime && !this.isOverTime(time))
			return true;
		else
			return false;
	}

	private synchronized boolean isOverTime(float time){
		if(time < this.endTime)
			return false;
		else
			return true;
	}

	private synchronized boolean objectIndexCheck(int object){
		if(object >= 0 && object < this.numObjects){
			return true;
		}
		else{
			logger.warning("Object index, " + object + ",out of range");
			return false;
		}
	}

	private synchronized void loadObjectsFromFile(String fileName){
		//open file
		BufferedReader input = null;
		logger.info("Opening " + fileName);
		try{
			input = new BufferedReader(new FileReader(fileName));
		} catch(FileNotFoundException e){
			System.out.println("ERROR: Object file not found");
			return;
		}
		logger.info(fileName + " Opened");
		
		try{
			//read number of objects in file
			this.numObjects = Integer.parseInt(input.readLine());
			logger.info(fileName + " contains " + this.numObjects);
			objects = new Object3D[this.numObjects];
			modifiedObjects = new Object3D[this.numObjects];
			for(int object = 0; object < this.numObjects; object++){
				//get object type
				String type = input.readLine();
				logger.info("Loading object " + object + " of type " + type);
				//get the correct factory type
				RenderableFactory rendFactory = 
					RenderableFactory.getFactory(type);
				//read next line which contains decorators and create
				String decos = input.readLine();
				this.objects[object] = Object3DFactory.newObject3D(type);
				if(this.objects[object] != null){
					this.objects[object].setModel(
						rendFactory.newRenderable(decos));
					this.objects[object].loadFromFile(input);
					this.objects[object].applyGravity(this.getGravity(),
						this.isGravityForce());
				}
			}
		} catch(IOException e){
			System.out.println("ERROR: Could not read object file.");
			return;
		}

		try{
			input.close();
			logger.info(fileName + " closed");
		} catch(IOException e){
			System.out.println("WARNING: Unable to close object file.");
		}
	}

	public Object clone() throws CloneNotSupportedException {
		throw new CloneNotSupportedException();
	}
}
