package rbs.sim;

import java.lang.CloneNotSupportedException;
import java.util.logging.Logger;
import javax.vecmath.Vector3f;
import javax.vecmath.Point3f;

public class Force implements Cloneable{
	private Vector3f force;
	private Point3f origin; //relative to the center of mass
	private float duration;

	private Vector3f acceleration;
	private boolean accelerationValid;
	private boolean directAcceleration;

	private Vector3f currentTransAcceleration;
	private Vector3f currentRotAcceleration;

	public Force(){
		this.force = new Vector3f();
		this.origin = new Point3f();
		this.duration = 0.0f;
		this.currentRotAcceleration = new Vector3f();
		this.currentTransAcceleration = new Vector3f();
		this.accelerationValid = false;
		this.directAcceleration = false;
	}

	public Force(Vector3f force, Point3f origin, float duration,
			boolean acceleration){
		this.force = new Vector3f(force);
		this.origin = new Point3f(origin);
		this.duration = duration;
		this.currentRotAcceleration = new Vector3f();
		this.currentTransAcceleration = new Vector3f();
		this.accelerationValid = acceleration;
		this.directAcceleration = false;
	}

	public Force(Vector3f direction, float magnitude, Point3f origin,
			float duration, boolean acceleration){
		Vector3f force = new Vector3f(direction);
		force.scale(magnitude);
		this.currentTransAcceleration = new Vector3f();
		this.currentRotAcceleration = new Vector3f();
		this.force = force;
		this.origin = new Point3f(origin);
		this.duration = duration;
		this.accelerationValid = acceleration;
		this.directAcceleration = false;
	}

	//only makes sense for NonImpulseForce
	public Force(Vector3f transAcceleration, Vector3f rotAcceleration){
		this.currentTransAcceleration = new Vector3f(transAcceleration);
		this.currentRotAcceleration = new Vector3f(rotAcceleration);
		this.directAcceleration = true;
		this.accelerationValid = false;
		this.force = new Vector3f();
		this.duration = 0.0f;
		this.origin = new Point3f();
	}

	protected void setDuration(float duration){
		this.duration = duration;
	}

	private float getDuration(){
		return this.duration;
	}

	protected void setForce(Vector3f force){
		this.force = force;
	}

	public Vector3f getForce(){
		return this.force;
	}

	private void setOrigin(Point3f origin){
		this.origin = origin;
	}

	public Point3f getOrigin(){
		return this.origin;
	}

	public Point3f getRelativeOrigin(Object3D obj){
		Point3f com = obj.getCenterOfMassWorld();
		com.add(this.origin);
		return com;
	}

	private void setCurrentTransAcceleration(Vector3f cta){
		this.currentTransAcceleration = cta;
	}

	private void setCurrentRotAcceleration(Vector3f cra){
		this.currentRotAcceleration = cra;
	}

	private Vector3f rotAcceleration(float timeDelta, Object3D obj){
		Vector3f moment = new Vector3f();

		Vector3f normal = new Vector3f(this.origin);

		if(normal.x == 0.0f && normal.y == 0.0f && normal.z == 0.0f)
			return moment;

		moment.cross(normal, this.force);

		obj.getLogger().fine("Moment on object = " + moment);
		//calculate angular acceleration for pitch yaw roll
		if(obj.getMassMomentOfInertiaPitch() != 0.0f)
			moment.x = moment.x / obj.getMassMomentOfInertiaPitch();
		if(obj.getMassMomentOfInertiaYaw() != 0.0f)
			moment.y = moment.y / obj.getMassMomentOfInertiaYaw();
		if(obj.getMassMomentOfInertiaRoll() != 0.0f)
			moment.z = moment.z / obj.getMassMomentOfInertiaRoll();

		moment.scale(timeDelta);
		obj.getLogger().fine("Angular Acceleration = " + moment);

		return moment;
	}

	private Vector3f transAcceleration(float timeDelta, Object3D obj){
		if(obj.getMass() == 0.0f)
			return new Vector3f();

		Vector3f accel = null;
		Vector3f normal = new Vector3f(this.origin);

		if(normal.x == 0.0f && normal.y == 0.0f && normal.z == 0.0f){
			//force is acting through the center of mass
			accel = new Vector3f(this.force);
		}
		else{
			//take care of translational effects
			normal.normalize();
			obj.getLogger().fine("Normalized action vector = " + normal);
			//compute force along normal
			float transMag = this.force.dot(normal);
			accel = new Vector3f(normal);
			accel.scale(transMag);
			obj.getLogger().fine("Force along normal = " + accel);
		}
		//apply the force
		//f = ma
		//a = f/m
		accel.scale(1.0f / obj.getMass());
		accel.scale(timeDelta);
		obj.getLogger().fine("Acceleration caused by applied force = " + accel);
		return accel;
	}

	public Vector3f currentTransAcceleration(){
		return this.currentTransAcceleration;
	}

	public Vector3f currentRotAcceleration(){
		return this.currentRotAcceleration;
	}

	//returns true if duration up
	public boolean apply(float timeDelta, Object3D obj){
		obj.getLogger().fine("Applying force of " + this.getForce() + "N at " +
			this.getOrigin());

		obj.getLogger().fine("Remaining Duration = " + this.duration);
		//figure out how long theforce is applied
		if(this.duration - timeDelta < 0.0f){
			timeDelta = this.duration;
		}

		if(this.accelerationValid){
			this.force.scale(obj.getMass());
			this.accelerationValid = false;
		}

		Vector3f transAccel = null;
		if(directAcceleration){
			transAccel = this.currentTransAcceleration;
		}
		else{
			transAccel = transAcceleration(timeDelta, obj);
			this.currentTransAcceleration.set(transAccel);
		}

		//change our velocity
		obj.getVelocity().add(transAccel);
		obj.getLogger().fine(
			"Velocity after force applied = " + obj.getVelocity());

		Vector3f rotAccel = null;
		if(directAcceleration){
			rotAccel = this.currentRotAcceleration;
		}
		else{
			rotAccel = rotAcceleration(timeDelta, obj);
			this.currentRotAcceleration.set(rotAccel);
		}

		//update angular velocity
		obj.getAngularVelocity().add(rotAccel);
		obj.getLogger().fine("Angular velocity after force applied = " +
			obj.getAngularVelocity());

		this.duration -= timeDelta;
		if(this.duration <= 0.0f)
			return true;
		else
			return false;
	}

	public Object clone() throws CloneNotSupportedException{
		Force copy = (Force)super.clone();
		copy.duration = this.duration;
		copy.setForce((Vector3f)this.getForce().clone());
		copy.setOrigin((Point3f)this.getOrigin().clone());
		copy.setCurrentRotAcceleration(
			(Vector3f)this.currentRotAcceleration().clone());
		copy.setCurrentTransAcceleration(
			(Vector3f)this.currentTransAcceleration().clone());
		return (Object)copy;
	}
}
