package rbs.render;

import java.util.logging.Logger;

import java.awt.Frame;
import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;
import javax.media.opengl.glu.GLUquadric;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLCanvas;
import javax.media.opengl.GLJPanel;
import javax.media.opengl.GLEventListener;
import javax.vecmath.Point3f;

import rbs.SimulationState;
import rbs.reel.StillFrame;

public class OpenGLRenderer extends Renderer implements GLEventListener{
	private static Logger logger = Logger.getLogger(
		OpenGLRenderer.class.getName());

	private Frame oglFrame;
	private GLCanvas oglCanvas;
	//private GLJPanel oglCanvas;
	private GLU oglGLU;
	private GLUquadric oglQuadric;
	private StillFrame currentFrame;

	private int x;
	private int y;
	private int width;
	private int height; 
	
	//Should I make a simple frustum/camera class?
	private float frustumLeft;
	private float frustumRight;
	private float frustumNear;
	private float frustumFar;

	private Point3f cameraLoc;
	private Point3f cameraLookAt;

	public OpenGLRenderer(){
		this.setState(RenderState.WINDOW_CREATED);
		this.oglFrame = new Frame();
		SimulationState.getInstance().configLocalLogger(logger);
	}

	public void setFrustum(float frontLeft, float frontRight, float near, 
		float far){
		if(this.getState() != RenderState.WINDOW_CREATED)
			return;

		this.frustumLeft = frontLeft;
		this.frustumRight = frontRight;
		this.frustumNear = near;
		this.frustumFar = far;
		this.setState(RenderState.FRUSTUM_INITIALIZED);
	}

	public void setCamera(Point3f location, Point3f lookAt){
		if(this.getState() != RenderState.FRUSTUM_INITIALIZED)
			return;

		this.cameraLoc = location;
		this.cameraLookAt = lookAt;
		this.setState(RenderState.CAMERA_INITIALIZED);
	}

	public void initializeAPI(int x, int y, int width, int height){
		if(this.getState() != RenderState.CAMERA_INITIALIZED)
			return;

		this.width = width;
		this.height = height;
		this.x = x;
		this.y = y;

		//for now we will just use the basic capabilities
		this.oglCanvas = new GLCanvas();
		this.oglCanvas.setSize(width, height);

		//add canvas to the frame, we need something to draw on
		this.oglFrame.add(this.oglCanvas);
		//we want this instance notified on gl events
		this.oglCanvas.addGLEventListener(this);

		this.setState(RenderState.API_INITIALIZED);
	}

	public void openWindow(String name){
		//make sure API specific initialization has happened
		if(this.getState() != RenderState.API_INITIALIZED){
			return;
		}

		this.oglFrame.setTitle(name);
		this.oglFrame.setSize(this.width, this.height);
		this.oglFrame.setLocation(this.x, this.y);
		this.oglFrame.setVisible(true);

		//the window is now open for business
		this.setState(RenderState.WINDOW_OPENED);
	}

	public void closeWindow(){
		//only close the window if it has already been opened
		if(this.getState() != RenderState.WINDOW_OPENED)
			return;
		this.oglFrame.setVisible(false);
		this.setState(RenderState.WINDOW_CLOSED);
	}
	
	public void destroyWindow(){
		//don't destroy the window unless it has already bee closed
		if(this.getState() != RenderState.WINDOW_CLOSED)
			return;
		this.oglFrame.dispose();
		this.setState(RenderState.WINDOW_DESTROYED);
	}

	public void render(StillFrame frame){
		//we can only render if the window is open
		if(this.getState() != RenderState.WINDOW_OPENED)
			return;

		if(frame != null)
			this.currentFrame = frame;
		this.oglCanvas.display();
	}

	//GLEventListener interface
	public void init(GLAutoDrawable drawable){
		GL gl = drawable.getGL();
		//clear the screen to black
		gl.glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
		gl.glShadeModel(GL.GL_SMOOTH);
		gl.glEnable(GL.GL_DEPTH_TEST);
		//create a quadric object, used to render spheres
		this.oglGLU = new GLU();
		this.oglQuadric = this.oglGLU.gluNewQuadric();
		oglGLU.gluQuadricNormals(this.oglQuadric, GL.GL_SMOOTH);
		oglGLU.gluQuadricTexture(this.oglQuadric, true);
		//initialize lighting
		float[] ambient = {0.5f, 0.5f, 0.5f, 1.0f};
		float[] diffuse = {1.0f, 1.0f, 1.0f, 1.0f};
		float[] position = {0.0f, 20.0f, 0.0f, 1.0f};
		gl.glLightfv(GL.GL_LIGHT1, GL.GL_AMBIENT, ambient, 0);
		gl.glLightfv(GL.GL_LIGHT1, GL.GL_DIFFUSE, diffuse, 0);
		gl.glLightfv(GL.GL_LIGHT1, GL.GL_POSITION, position, 0);
		gl.glEnable(GL.GL_LIGHT1);
		gl.glEnable(GL.GL_LIGHTING);
		gl.glColorMaterial(GL.GL_FRONT_AND_BACK, GL.GL_DIFFUSE);
		gl.glEnable(GL.GL_COLOR_MATERIAL);
		//set up texturing
		gl.glTexEnvi(GL.GL_TEXTURE_ENV, GL.GL_TEXTURE_ENV_MODE, GL.GL_BLEND);
	}

	public void display(GLAutoDrawable drawable){
		GL gl = drawable.getGL();
		gl.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT);

		//set up the camera
		gl.glLoadIdentity();
		this.oglGLU.gluLookAt(this.cameraLoc.x, this.cameraLoc.y,
			this.cameraLoc.z, this.cameraLookAt.x, this.cameraLookAt.y,
			this.cameraLookAt.z, 0.0d, 1.0d, 0.0d);

		if(this.currentFrame != null){
			//render contents of frame
			int numModels = this.currentFrame.numModels();
			for(int obj = 0; obj < numModels; obj++){
				gl.glPushMatrix();
				Renderable object = this.currentFrame.nextModel();
				object.renderOpenGL(gl, this.oglGLU, this.oglQuadric);
				gl.glPopMatrix();
			}
		}
		gl.glFlush();
	}

	//Currently not implemented but part of GLEventListener interface
	public void displayChanged(GLAutoDrawable drawable, boolean modeChanged,
		boolean deviceChanged) {
	}

	public void reshape(GLAutoDrawable drawable, int x,int y, int width, 
		int height){
		//logger.entering(OpenGLRenderer.class.getName(), "reshape");
		GL gl = drawable.getGL();

		//set the new viewport
		gl.glViewport(0, 0, width, height);
		logger.info("Viewport set x = 0, y = 0, width = " + width + 
			", height = " + height);

		float h = (float)height / (float)width;
		logger.info("Frustum near plane height computed to be " + h);

		//determins how the object is drawn on screen
		gl.glMatrixMode(GL.GL_PROJECTION);
		gl.glLoadIdentity();
		//create viewing frustum
		gl.glFrustum(this.frustumLeft, this.frustumRight, -h, h,
			this.frustumNear, this.frustumFar);
		logger.info("Frustum set left = " + this.frustumLeft + ", right = " +
			this.frustumRight + ", top = " + h + ", bottom = " + -h + 
			", near = " + this.frustumNear + ", far = " + this.frustumFar);

		gl.glMatrixMode(GL.GL_MODELVIEW);
		//logger.exiting(OpenGLRenderer.class.getName(), "reshape");
	}
}
