package rbs.config;

import java.util.HashMap;
import java.util.ArrayList;
import java.util.Scanner;
import java.lang.IllegalArgumentException;
import java.lang.IllegalAccessException;
import java.lang.IllegalStateException;
import java.lang.ClassNotFoundException;
import java.lang.InstantiationException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;

public class ConfigFile{
	private final static String CONSTRUCTOR_ARG_TYPE = "java.lang.String";

	private String filename;
	private HashMap<String, String> typeMap;
	private ArrayList<Setting> settings;

	public ConfigFile(String filename){
		this.filename = filename;
		this.typeMap = new HashMap<String, String>();
		this.settings = new ArrayList<Setting>();
	}

	public ConfigFile(String filename, int numSettings){
		this.filename = filename;
		this.typeMap = new HashMap<String, String>(numSettings);
		this.settings = new ArrayList<Setting>(numSettings);
	}

	public void read(){
		//attempt to open and read file
		Scanner input = null;
		try{
			input = new Scanner(new BufferedReader(new FileReader(filename)));
			//read a name value pair
			while(input.hasNext()){
				String settingName = input.next();
				Setting setting = getSetting(settingName);
				if(setting == null){
					System.out.println(
						"WARNING: Unknown Setting, " + settingName);
					continue;
				}
				if(input.hasNext()){
					Class typeClass = 
								Class.forName(this.typeMap.get(settingName));
					setSetting(typeClass, setting, input.next());
				}
				else{
					System.out.println("WARNING: Setting missing value.");
				}
			}
		} catch(FileNotFoundException e){
			System.out.print("WARNING: Configuration file not found, ");
			System.out.println("using default settings");
		} catch(IllegalStateException e){
			System.out.print("WARNING: Error parsing configuration file, ");
			System.out.println("using default settings");
		} catch(ClassNotFoundException e){
			System.out.print("WARNING: Could not load setting type, ");
			System.out.println("using default value for setting");
		}
		finally{
			if(input != null){
				input.close();
			}
		}
	}

	//go through each setting writing name and value
	public void write(){
		BufferedWriter output = null;
		try{
			output = new BufferedWriter(new FileWriter(filename));
			for(int setting = 0; setting < this.settings.size(); setting++){
				String settingName = this.settings.get(setting).getName();
				output.write(settingName + "\t");
				Class typeClass = Class.forName(this.typeMap.get(settingName));
				this.writeSetting(typeClass,this.settings.get(setting), output);
			}
			output.close();
		} catch(IOException e){
			System.out.println("ERROR: Could not write to file.");
		} catch(ClassNotFoundException e){
			System.out.print("WARNING: Could not load setting type, ");
			System.out.println("using default value for setting");
		}
	}

	@SuppressWarnings("unchecked")
	private <T> void writeSetting(Class T, Setting setting, 
			BufferedWriter output) throws IOException{
		PrimativeSetting<T> ps = (PrimativeSetting<T>)setting;
		output.write(ps.getValue() + "\n");
	}

	public double getDouble(String settingName) throws IllegalArgumentException{
		Double dbl = this.getValue(Double.class.getName(), settingName);
		if(dbl != null)
			return dbl.doubleValue();
		else
			throw new IllegalArgumentException("Unknown Setting");
	}

	public byte getByte(String settingName) throws IllegalArgumentException{
		Byte theByte = this.getValue(Byte.class.getName(), settingName);
		if(theByte != null)
			return theByte.byteValue();
		else
			throw new IllegalArgumentException("Unknown Setting");
	}

	public boolean getBoolean(String settingName) 
										throws IllegalArgumentException{
		Boolean bool = this.getValue(Boolean.class.getName(), settingName);
		if(bool != null)
			return bool.booleanValue();
		else
			throw new IllegalArgumentException("Unknown Setting");
	}

	public int getInteger(String settingName) throws IllegalArgumentException{
		Integer integer = this.getValue(Integer.class.getName(), settingName);
		if(integer != null)
			return integer.intValue();
		else
			throw new IllegalArgumentException("Unknown Setting");
	}

	public float getFloat(String settingName) throws IllegalArgumentException{
		Float flt = this.getValue(Float.class.getName(), settingName);
		if(flt != null)
			return flt.floatValue();
		else
			throw new IllegalArgumentException("Unknown Setting");
	}

	public String getString(String settingName) throws IllegalArgumentException{
		String string = this.getValue(String.class.getName(), settingName);
		if(string != null)
			return string;
		else
			throw new IllegalArgumentException("Unknown Setting");
	}

	protected void addInteger(String settingName){
		this.addSetting(settingName, new Integer(0));
	}

	protected void addBoolean(String settingName){
		this.addSetting(settingName, new Boolean(false));
	}

	protected void addByte(String settingName){
		this.addSetting(settingName, new Byte((byte)0));
	}

	protected void addDouble(String settingName){
		this.addSetting(settingName, new Double(0.0d));
	}

	protected void addFloat(String settingName){
		this.addSetting(settingName, new Float(0.0f));
	}

	protected void addString(String settingName){
		this.addSetting(settingName, new String(""));
	}

	private <T> void addSetting(String settingName, T initialValue){
		this.typeMap.put(settingName, initialValue.getClass().getName());
		PrimativeSetting<T> ps = new PrimativeSetting<T>(settingName);
		ps.setValue(initialValue);
		this.settings.add(ps);
	}

	//creates a new object of type T and changes value of proper setting to T
	@SuppressWarnings("unchecked")
	private <T> void setSetting(Class T, Setting setting, String value){
		//create a new object of type T
		T objValue = null;
		try{
			Constructor con = T.getConstructor(
					Class.forName(ConfigFile.CONSTRUCTOR_ARG_TYPE));
			objValue = (T)con.newInstance(value);
		} catch(ClassNotFoundException e){
			System.out.println("Could not load String class.");
		} catch(NoSuchMethodException e){
			System.out.println("Setting value does not have a constuctor");
			System.out.println(" which takes a String as an argument.");
		} catch(InstantiationException e){
			System.out.println("Could not Instantiate setting value.");
		} catch(IllegalAccessException e){
			System.out.println("Found Constructor is not accessible.");
		} catch(InvocationTargetException e){
			System.out.println("Setting value's constructor threw exception.");
		}

		if(objValue == null){
			System.out.print("WARNING: Could not load setting type, ");
			System.out.println("using default value for setting");
			return;
		}

		//cast the setting as a primitive setting
		PrimativeSetting<T> ps = (PrimativeSetting<T>)setting;
		ps.setValue(objValue);
	}

	@SuppressWarnings("unchecked")
	private <T> T getValue(String typeName, String settingName){
		if(this.typeMap.containsKey(settingName) &&
			this.typeMap.get(settingName).equals(typeName)){
			Setting setting = this.getSetting(settingName);
			if(setting != null){
				PrimativeSetting<T> ps = (PrimativeSetting<T>)setting;
				return ps.getValue();
			}
			else{
				System.out.print("ERROR: Configuration Settings are not");
				System.out.println(" setup properly");
			}
		}
		else{
			System.out.print("WARNING: Unknown setting, or incorrect setting");
			System.out.println(" type");
		}
		return null;
	}

	private Setting getSetting(String settingName){
		for(int index = 0; index < this.settings.size(); index++){
			if(settings.get(index).getName().equalsIgnoreCase(settingName))
				return this.settings.get(index);
		}
		return null;
	}
}
