package rbs;

import java.lang.IllegalArgumentException;
import java.lang.CloneNotSupportedException;
import java.util.logging.Logger;
import java.util.logging.Handler;
import java.util.logging.FileHandler;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.Filter;
import java.util.logging.LogRecord;
import java.util.logging.SimpleFormatter;
import java.io.IOException;

import rbs.ui.Playback;

public class SimulationState implements Playback{
	private static final String LOG_FILE = "rbs.log";
	private static final String ROOT_LOGGER_NAME = "rbs";
	private static final String DEFAULT_DEBUG = "WARNING";

	public static final String DEBUG_SETTING = "debug";

	private static SimulationState instance;
	private String debug;
	private boolean finished;
	private boolean paused;
	private boolean ff;
	private boolean rew;
	private boolean begin;
	private boolean end;
	private float jumpTime;
	private boolean friction;
	
	//logging objects
	private FileHandler logfile;
	private ConsoleHandler console;
	
	private SimulationState(){
		finished = false;	
		paused = true;
		debug = "";
		console = new ConsoleHandler();
		try{
			logfile = new FileHandler(SimulationState.LOG_FILE);
		} catch(IOException e){
			System.out.println("WARNING: Could not open log file");
		}
	}

	public static synchronized SimulationState getInstance(){
		if(instance == null){
			instance = new SimulationState();
		}
		return instance;
	}

	public Object clone() throws CloneNotSupportedException{
		throw new CloneNotSupportedException();
	}

	public synchronized void readConfig(RbsConfig config){
		try{
			debug = config.getString(SimulationState.DEBUG_SETTING);
		} catch(IllegalArgumentException e){
			System.out.print("ERROR: Could not initialize simulation state, ");
			System.out.println("setting does not exist.");
		}

		try{
			console.setFilter(new ConsoleFilter(Level.parse(this.debug)));
		} catch(IllegalArgumentException e){
			System.out.println(
				"WARNING: Unknown debug level set in config.txt");
		}
	}

	public synchronized void configLocalLogger(Logger logger){
		//remove all handlers for the given logger
		Handler[] handlers = logger.getHandlers();
		for(int handler = 0; handler < handlers.length; handler++){
			logger.removeHandler(handlers[handler]);
		}

		//add a file handler
		if(this.logfile != null)
			logger.addHandler(this.logfile);

		//add a console handler
		if(console.getFilter() == null){
			console.setFilter(new ConsoleFilter(Level.parse(DEFAULT_DEBUG)));
		}
		logger.addHandler(this.console);

		//set to finest level
		logger.setLevel(Level.FINEST);
		logger.setUseParentHandlers(false);
	}

	public synchronized String getDebug(){
		return this.debug;
	}

	public synchronized boolean isPaused(){
		return this.paused;
	}

	public synchronized void setPaused(boolean paused){
		this.paused = paused;
	}

	public synchronized boolean isFinished(){
		return this.finished;
	}

	public synchronized void setFinished(boolean finished){
		this.finished = finished;
	}

	public synchronized boolean isFf(){
		return this.ff;
	}

	public synchronized void setFf(boolean ff){
		this.ff = ff;
	}

	public synchronized boolean isRew(){
		return this.rew;
	}

	public synchronized void setRew(boolean rew){
		this.rew = rew;
	}

	public synchronized boolean isBegin(){
		return this.begin;
	}

	public synchronized void setBegin(boolean begin){
		this.begin = begin;
	}

	public synchronized boolean isEnd(){
		return this.end;
	}

	public synchronized void setEnd(boolean end){
		this.end = end;
	}

	public synchronized void setJumpTime(float time){
		this.jumpTime = time;
	}

	public synchronized float getJumpTime(){
		return this.jumpTime;
	}

	public synchronized void play(){
		this.setPaused(false);
	}

	public synchronized void pause(){
		this.setPaused(true);
	}

	public synchronized void stop(){
		this.setPaused(true);
		this.begin();
	}

	public synchronized void rewind(){
		this.setRew(true);	
	}

	public synchronized void ff(){
		this.setFf(true);
	}

	public synchronized void begin(){
		this.setBegin(true);	
	}

	public synchronized void end(){
		this.setEnd(true);
	}
	
	public synchronized boolean isFrictionEnabled(){
		return this.friction;
	}

	public synchronized void setFrictionEnabled(boolean friction){
		this.friction = friction;
	}
}

class ConsoleFilter implements Filter{
	private Level minLevel;

	public ConsoleFilter(Level minLevel){
		this.minLevel = minLevel;
	}

	public boolean isLoggable(LogRecord record){
		if(record.getLevel().intValue() >= minLevel.intValue())
			return true;
		else
			return false;
	}
}
