package rbs;

import java.util.ArrayList;
import java.util.logging.Logger;

import rbs.render.RendererThread;
import rbs.render.RendererThreadFactory;
import rbs.reel.FrameReel;
import rbs.reel.FrameReelFactory;
import rbs.sim.Object3D;
import rbs.sim.Sphere;
import rbs.sim.SimulationThread;
import rbs.sim.SimulationThreadFactory;
import rbs.sim.MaterialStore;
import rbs.ui.CommandLineInterface;
import rbs.ui.PlaybackGUI;
import rbs.ui.GraphicalInterface;
import rbs.ui.SerializableReel;

import javax.vecmath.Vector3f;

public class RigidBodySimulator{
	private static final String CONFIG_FILE = "etc/config.txt";
	private static Logger logger = 
			Logger.getLogger(RigidBodySimulator.class.getName());
	private static ArrayList<GraphicalInterface> guis = 
		new ArrayList<GraphicalInterface>();

	private RbsConfig config;
	private FrameReel reel;
	private RendererThread renderer;
	private SimulationThread sim;
	private CommandLineInterface cli;

	public RigidBodySimulator(){
		config = new RbsConfig(CONFIG_FILE);
		SimulationState.getInstance().configLocalLogger(logger);
	}

	public void startup(){
		System.out.println("=======================================");
		System.out.println("-Loading configuration file");
		config.read();
		System.out.println("-Setting Initial Simulation State");
		this.initSimState();

		logger.info("Rigid Body Simulator Initialization Started");

		//interfaces
		System.out.println("-Initializing User Interface");
		logger.info("User Interface Initialization Started");
		this.initInterface();
		logger.info("User Interface Initialization Complete");

		//reel init
		logger.info("Reel Initialization Started");
		System.out.println("-Creating Frame Reel");
		this.initReel();
		logger.info("Reel Initialization Complete");

		//renderer init
		logger.info("Rendering Initialization Started");
		System.out.println("-Initializing Rendering Subsystem");
		this.initRenderer();
		logger.info("Rendering Initialization Complete");

		//simulation init
		logger.info("3D Simulation Initialization Started");
		System.out.println("-Initializing 3D Simulation");
		this.initSimulation();
		logger.info("3D Simulation Initialization Complete");

		System.out.println("=======================================\n");
		System.out.println("Type 'help' for list of commands");

		logger.info("Rigid Body Simulator Initialization Complete");

		//start the command line interface if it exists
		if(cli != null)
			cli.start();
	}

	private void initInterface(){
		if(this.config.getBoolean(UI_COMMAND_LINE)){
			cli = new CommandLineInterface();
			if(this.config.getBoolean(UI_PLAYBACK_CLI)){
				cli.addPlaybackCommands(SimulationState.getInstance());
			}
			cli.initInterface();
		}

		if(this.config.getBoolean(UI_PLAYBACK_GUI)){
			PlaybackGUI pbg = new PlaybackGUI(SimulationState.getInstance());
			pbg.initInterface();
			pbg.start();
			guis.add(pbg);
		}
	}

	private void initSimState(){
		SimulationState.getInstance().readConfig(config);
	}

	private void initSimulation(){
		String materialFile = this.config.getString(SIM_MATERIAL_FILE_SETTING);
		MaterialStore.getInstance().loadFromFile(materialFile);
		String objFile = this.config.getString(SIM_OBJECT_FILE_SETTING);
		float timeStepSize = this.config.getFloat(SIM_TIMESTEP_SIZE_SETTING);
		int numTimeSteps = this.config.getInteger(SIM_NUM_TIMESTEPS_SETTING);
		SimulationState.getInstance().setJumpTime(2.0f * timeStepSize);
		boolean pauseOnCollision = this.config.getBoolean(
			SIM_PAUSE_ON_COLLISION);
		float gravityX = this.config.getFloat(SIM_GRAVITY_X_SETTING);
		float gravityY = this.config.getFloat(SIM_GRAVITY_Y_SETTING);
		float gravityZ = this.config.getFloat(SIM_GRAVITY_Z_SETTING);
		boolean gravityForce = this.config.getBoolean(SIM_GRAVITY_FORCE);
		sim = SimulationThreadFactory.newSimulationThread("react_time_step",
			objFile, numTimeSteps, timeStepSize, pauseOnCollision,
			new Vector3f(gravityX, gravityY, gravityZ), gravityForce);
		sim.setReel(this.reel);
		boolean friction = this.config.getBoolean(SIM_FRICTION);
		SimulationState.getInstance().setFrictionEnabled(friction);
	}

	private void initReel(){
		//see what type of reel to create
		String type = config.getString(REEL_TYPE_SETTING);
		this.reel = FrameReelFactory.newFrameReel(type);

		if(this.config.getBoolean(UI_SERIAL_REEL_CLI) && 
				this.reel instanceof SerializableReel){
			cli.addSerializableReelCommands((SerializableReel)this.reel);
		}
	}

	private void initRenderer(){
		int numRenderers = this.config.getInteger(NUM_RENDERERS_SETTING);
		logger.config("Creating " + numRenderers + " renderers");

		if(numRenderers <= 0)
			return;

		//TODO: Should have a file setting for renderer type
		renderer = RendererThreadFactory.newRendererThread("keyframe", "rend");
		renderer.setReel(this.reel);

		String[] types = new String[numRenderers];
		String[] names = new String[numRenderers];
		float[] cameraX = new float[numRenderers];
		float[] cameraY = new float[numRenderers];
		float[] cameraZ = new float[numRenderers];
		float[] fleft = new float[numRenderers];
		float[] fright = new float[numRenderers];
		float[] fnear = new float[numRenderers];
		float[] ffar = new float[numRenderers];
		float[] lookatX = new float[numRenderers];
		float[] lookatY = new float[numRenderers];
		float[] lookatZ = new float[numRenderers];

		logger.info("Reading renderer information from config file");
		for(int index = 0; index < numRenderers; index++){
			types[index] = this.config.getString(
				String.format(RENDER_TYPE_FORMAT, index));
			names[index] = String.format(RigidBodySimulator.RENDER_NAME_FORMAT,
					index);
			fleft[index] = this.config.getFloat(String.format(
				FRUSTUM_LEFT_FORMAT, index));
			fright[index] = this.config.getFloat(String.format(
				FRUSTUM_RIGHT_FORMAT, index));
			fnear[index] = this.config.getFloat(String.format(
				FRUSTUM_NEAR_FORMAT, index));
			ffar[index] = this.config.getFloat(String.format(
				FRUSTUM_FAR_FORMAT, index));
			cameraX[index] = this.config.getFloat(String.format(
				CAMERA_X_FORMAT, index));
			cameraY[index] = this.config.getFloat(String.format(
				CAMERA_Y_FORMAT, index));
			cameraZ[index] = this.config.getFloat(String.format(
				CAMERA_Z_FORMAT, index));
			lookatX[index] = this.config.getFloat(String.format(
				LOOKAT_X_FORMAT, index));
			lookatY[index] = this.config.getFloat(String.format(
				LOOKAT_Y_FORMAT, index));
			lookatZ[index] = this.config.getFloat(String.format(
				LOOKAT_Z_FORMAT, index));
		}
		int width = this.config.getInteger(R_WIDTH_SETTING);
		int height = this.config.getInteger(R_HEIGHT_SETTING);

		logger.info("Finished reading renderer information from config file");

		renderer.createWindows(numRenderers, types);

		//setup the camera and frustum
		renderer.setFrustums(fleft, fright, fnear, ffar);
		renderer.setCameras(cameraX, cameraY, cameraZ, lookatX, lookatY,
			lookatZ);

		renderer.initializeAPIs(5, 100, width, height);
		
		renderer.openWindows(names);
	}

	public void shutdown(){
		for(int index = 0; index < guis.size(); index++){
			guis.get(index).close();
		}
	}

	public static void main(String[] args){
		System.out.println("Starting Rigid Body Simulator");
		RigidBodySimulator rbs = new RigidBodySimulator();
		rbs.startup();
		rbs.shutdown();
	}

	////////////////////////////////////////////////////////////////////////////
	//Formats
	////////////////////////////////////////////////////////////////////////////
	private static final String RENDER_TYPE_FORMAT = "renderer%d_type";
	private static final String FRUSTUM_LEFT_FORMAT = "renderer%d_frustum_left";
	private static final String FRUSTUM_RIGHT_FORMAT="renderer%d_frustum_right";
	private static final String FRUSTUM_NEAR_FORMAT = "renderer%d_frustum_near";
	private static final String FRUSTUM_FAR_FORMAT = "renderer%d_frustum_far";
	private static final String CAMERA_X_FORMAT = "renderer%d_camera_x";
	private static final String CAMERA_Y_FORMAT = "renderer%d_camera_y";
	private static final String CAMERA_Z_FORMAT = "renderer%d_camera_z";
	private static final String LOOKAT_X_FORMAT = "renderer%d_lookat_x";
	private static final String LOOKAT_Y_FORMAT = "renderer%d_lookat_y";
	private static final String LOOKAT_Z_FORMAT = "renderer%d_lookat_z";
	private static final String RENDER_NAME_FORMAT = "Renderer %d";
	////////////////////////////////////////////////////////////////////////////
	//SETTINGS
	////////////////////////////////////////////////////////////////////////////
	//Renderer Settings
	public static final String NUM_RENDERERS_SETTING= "num_renderers";
	public static final String R0_TYPE_SETTING 		= "renderer0_type";
	public static final String R1_TYPE_SETTING 		= "renderer1_type";
	public static final String R2_TYPE_SETTING 		= "renderer2_type";
	public static final String R3_TYPE_SETTING 		= "renderer3_type";

	public static final String R_WIDTH_SETTING 		= "renderer_width";
	public static final String R_HEIGHT_SETTING 	= "renderer_height";

	public static final String R0_FLEFT_SETTING		= "renderer0_frustum_left";
	public static final String R1_FLEFT_SETTING		= "renderer1_frustum_left";
	public static final String R2_FLEFT_SETTING		= "renderer2_frustum_left";
	public static final String R3_FLEFT_SETTING		= "renderer3_frustum_left";

	public static final String R0_FRIGHT_SETTING	= "renderer0_frustum_right";
	public static final String R1_FRIGHT_SETTING	= "renderer1_frustum_right";
	public static final String R2_FRIGHT_SETTING	= "renderer2_frustum_right";
	public static final String R3_FRIGHT_SETTING	= "renderer3_frustum_right";

	public static final String R0_FNEAR_SETTING		= "renderer0_frustum_near";
	public static final String R1_FNEAR_SETTING		= "renderer1_frustum_near";
	public static final String R2_FNEAR_SETTING		= "renderer2_frustum_near";
	public static final String R3_FNEAR_SETTING		= "renderer3_frustum_near";

	public static final String R0_FFAR_SETTING		= "renderer0_frustum_far";
	public static final String R1_FFAR_SETTING		= "renderer1_frustum_far";
	public static final String R2_FFAR_SETTING		= "renderer2_frustum_far";
	public static final String R3_FFAR_SETTING		= "renderer3_frustum_far";

	public static final String R0_CAMERAX_SETTING	= "renderer0_camera_x";
	public static final String R1_CAMERAX_SETTING	= "renderer1_camera_x";
	public static final String R2_CAMERAX_SETTING	= "renderer2_camera_x";
	public static final String R3_CAMERAX_SETTING	= "renderer3_camera_x";

	public static final String R0_CAMERAY_SETTING	= "renderer0_camera_y";
	public static final String R1_CAMERAY_SETTING	= "renderer1_camera_y";
	public static final String R2_CAMERAY_SETTING	= "renderer2_camera_y";
	public static final String R3_CAMERAY_SETTING	= "renderer3_camera_y";

	public static final String R0_CAMERAZ_SETTING	= "renderer0_camera_z";
	public static final String R1_CAMERAZ_SETTING	= "renderer1_camera_z";
	public static final String R2_CAMERAZ_SETTING	= "renderer2_camera_z";
	public static final String R3_CAMERAZ_SETTING	= "renderer3_camera_z";

	public static final String R0_LOOKATX_SETTING	= "renderer0_lookat_x";
	public static final String R1_LOOKATX_SETTING	= "renderer1_lookat_x";
	public static final String R2_LOOKATX_SETTING	= "renderer2_lookat_x";
	public static final String R3_LOOKATX_SETTING	= "renderer3_lookat_x";
	
	public static final String R0_LOOKATY_SETTING	= "renderer0_lookat_y";
	public static final String R1_LOOKATY_SETTING	= "renderer1_lookat_y";
	public static final String R2_LOOKATY_SETTING	= "renderer2_lookat_y";
	public static final String R3_LOOKATY_SETTING	= "renderer3_lookat_y";

	public static final String R0_LOOKATZ_SETTING	= "renderer0_lookat_z";
	public static final String R1_LOOKATZ_SETTING	= "renderer1_lookat_z";
	public static final String R2_LOOKATZ_SETTING	= "renderer2_lookat_z";
	public static final String R3_LOOKATZ_SETTING	= "renderer3_lookat_z";
	//Reel Settings
	public static final String REEL_TYPE_SETTING			= "reel_type";
	//Simulation Settings
	public static final String SIM_OBJECT_FILE_SETTING		= "object_file";
	public static final String SIM_MATERIAL_FILE_SETTING	= "material_file";
	public static final String SIM_NUM_TIMESTEPS_SETTING	= "num_timesteps";
	public static final String SIM_TIMESTEP_SIZE_SETTING	= "timestep_size";
	public static final String SIM_PAUSE_ON_COLLISION	= "pause_on_collision";
	public static final String SIM_GRAVITY_X_SETTING			= "gravity_x";
	public static final String SIM_GRAVITY_Y_SETTING			= "gravity_y";
	public static final String SIM_GRAVITY_Z_SETTING			= "gravity_z";
	public static final String SIM_GRAVITY_FORCE			= "gravity_force";
	public static final String SIM_FRICTION					= "friction";
	//interface settings
	public static final String UI_COMMAND_LINE				= "command_line";
	public static final String UI_PLAYBACK_CLI			= "playback_cli";
	public static final String UI_PLAYBACK_GUI			= "playback_gui";
	public static final String UI_SERIAL_REEL_CLI		= "serial_reel_cli";
	public static final String UI_SERIAL_REEL_GUI		= "serial_reel_gui";
	////////////////////////////////////////////////////////////////////////////
	//END SETTINGS
	////////////////////////////////////////////////////////////////////////////
}
