package main

import (
	"os"
	"fmt"
	"nsutil"
	"saxutil"
	"github.com/ziutek/mymysql/mysql"
	_ "github.com/ziutek/mymysql/native"
	"bitbucket.org/binet/go-gnuplot/pkg/gnuplot"
)

func main() {

	if len(os.Args) < 2 {
		fmt.Fprintf(os.Stderr, "Usage: %s <config file>\n", os.Args[0])
		os.Exit(1)
	}

	config, err := nsutil.Read_config(os.Args[1])
	if err != nil {
		fmt.Fprintf(os.Stderr, "Error reading configuration file.\n")
	}

	db := mysql.New("tcp", "", config.Sql_host+":"+config.Sql_port,
					config.Sql_user, config.Sql_passwd, config.Sql_db)

	err = db.Connect()
	if err != nil {
		fmt.Fprintf(os.Stderr, "Could not connect to database.\n")
		os.Exit(1)
	}

	query_str := fmt.Sprintf("select distinct host from %s;",
								config.Sql_table)

	hosts, hostres, err := db.Query(query_str)

	for _, hostrow := range hosts {
		host := hostrow.Str(hostres.Map("host"))
		fmt.Printf("Handling %s\n", host);

		query_str = fmt.Sprintf("select distinct label from %s "+
							"where host=\"%s\";", config.Sql_table, host)

		properties, propres, _ := db.Query(query_str)

		for _, proprow := range properties {
			property := proprow.Str(propres.Map("label"))
			fmt.Printf("\t%s\n", property);

			html, err := os.Create(nsutil.Pre_name(host, property)+".html")
			if err != nil {
				fmt.Fprintf(os.Stderr, "Could not create %s\n",
						nsutil.Pre_name(host, property)+".html")
				os.Exit(1)
			}

			fmt.Fprintf(html, "<html>\n<body>\n")
			fmt.Fprintf(html, "<form action="+
				"\"http://super.ece.engr.uky.edu:8092/ns-bin/setbase.py\""+
				" method=\"get\">\n")
			fmt.Fprintf(html, "<input type=\"hidden\" name=\"host\" "+
						"value=\"%s\">\n", host)
			fmt.Fprintf(html, "<input type=\"hidden\" name=\"property\" "+
						"value=\"%s\">\n", property)

			for i := 0; d2s(i) < config.DB_hist; i += 1 {

				query_str = fmt.Sprintf("select data, ctime from %s "+
					"where host=\"%s\" and label=\"%s\" "+
					"and ctime < date_sub(now(), interval %d second) "+
					"and ctime > date_sub(now(), interval %d second) "+
					"order by ctime asc;",
					config.Sql_table, host, property, d2s(i), d2s(i+1))
				/* query data */
				rows, res, err := db.Query(query_str)

				values, times := saxutil.Arrays_from_rows(rows, res)

				prefix := fmt.Sprintf("%s_%d",
						nsutil.Pre_name(host, property), i)
				dump(values, times, "assets/"+prefix+".dat")
				/* plot data in <host>_<property>_day<i>.png */

				plotter, err := gnuplot.NewPlotter("", true, false)
				if err != nil {
					fmt.Fprintf(os.Stderr, "Error getting plotter.\n")
					fmt.Fprintf(os.Stderr, "Exiting...\n")
					os.Exit(1)
				}
				plotter.CheckedCmd("set terminal png small size 1000,200")
				plotter.CheckedCmd("set size 1,1")
				plotter.CheckedCmd("set xdata time")
				plotter.CheckedCmd("set timefmt \"%%s\"")
				plotter.CheckedCmd("set format x \"%%b %%d %%R\"")

				err = plotter.SetLabels("Time", "Sample value")
				plotter.CheckedCmd("set output \"assets/%s.png\"", prefix)
				plotter.CheckedCmd(
					"plot \"assets/%s.dat\" u 2:1 t 'day %d' w lines", prefix, i)
				plotter.Close()

				fmt.Fprintf(html,
					"<img src=\"assets/%s.png\">\n", prefix)
				fmt.Fprintf(html, "<input type=\"checkbox\" "+
						"name=\"baseline\" value=\"%d\">\n", i)
				fmt.Fprintf(html, "<hr>\n")

			} /* for i < config.DB_hist */

			fmt.Fprintf(html, "<br>\n")
			fmt.Fprintf(html, "<input type=\"submit\" name=\"submit\">\n")
			fmt.Fprintf(html, "</form>\n</body>\n</html>\n")
		} /* range over labels */
	} /* range over hosts */

}

func d2s(days int) int {
	return days * 60 * 60 * 24;
}

func dump(values []float64, times []int, filename string) {

	fout, err := os.Create(filename)
	if err != nil {
		fmt.Fprintf(os.Stderr, "Could not create %s.\n", filename)
		os.Exit(1)
	}

	length := len(values)
	if length > len(times) {
		length = len(times)
	}

	for i := 0; i < length; i++ {
		fmt.Fprintf(fout, "%f\t%d\n", values[i], times[i])
	}
}
