package main

import (
	"os"
	"fmt"
	"nsutil"
	"saxutil"
	"strings"
	"strconv"
	"os/exec"
	"bitbucket.org/binet/go-gnuplot/pkg/gnuplot"
	"github.com/ziutek/mymysql/mysql"
	_ "github.com/ziutek/mymysql/native"
)

func pre_name(host, property string) (name string) {
	name = host+property
	name = strings.Replace(name, " ", "", -1)
	name = strings.Replace(name, "\t", "", -1)
	return
}

func main() {

	if len(os.Args) <  4 {
		fmt.Fprintf(os.Stderr,
		"Usage: %s <config file> <width> <height>\n",
		os.Args[0])
		os.Exit(0)
	}

	config, err := nsutil.Read_config(os.Args[1])

	if err != nil {
		fmt.Fprintf(os.Stderr, "Error reading configuration file.\n")
	}

	var values []float64
	var times []int
	var width, height int64

	width, _ = strconv.ParseInt(os.Args[2], 10, 64)
	height, _ = strconv.ParseInt(os.Args[3], 10, 64)

	db := mysql.New("tcp", "", config.Sql_host+":"+config.Sql_port,
			config.Sql_user, config.Sql_passwd, config.Sql_db)

	err = db.Connect()
	if err != nil {
		fmt.Fprintf(os.Stderr, "Could not connect to database.\n")
		os.Exit(1)
	}

	for _, host := range saxutil.Get_hosts(db, config) {
		fmt.Fprintf(os.Stderr, "Handling %s\n", host)

		for _, property := range saxutil.Get_labels(db, host, config) {
			fmt.Fprintf(os.Stderr, "\t%s\n", property)

			fname := fmt.Sprintf("%s", pre_name(host, property))

			rows, res, err := nsutil.Get_data(host, property, config)
			values, times = saxutil.Arrays_from_rows(rows, res)


			fout, err := os.Create(fname+".dat")
			if err != nil {
				fmt.Fprintf(os.Stderr, "Could not create %s.dat\n", fname)
				os.Exit(1)
			}

			for i := 0; i < len(values); i++ {

				fmt.Fprintf(fout, "%f\t%d\n", values[i], times[i])
			}

			fout.Close()

			plotter, err := gnuplot.NewPlotter("", true, false)
			if err != nil {
				fmt.Fprintf(os.Stderr, "Error getting plotter.\n")
				fmt.Fprintf(os.Stderr, "Exiting...\n")
				os.Exit(0)
			}

			plotter.CheckedCmd("set terminal png small size %d,%d", width, height)
			plotter.CheckedCmd("set size 1,1")
			plotter.CheckedCmd("set xdata time")
			plotter.CheckedCmd("set timefmt \"%%s\"")
			plotter.CheckedCmd("set format x \"%%b %%d %%R\"")

			err = plotter.SetLabels("Time", "Sample value")
			plotter.CheckedCmd("set output \"%s.png\"", fname)
			plotter.CheckedCmd("plot \"%s.dat\" u 2:1 t 'values' w lines lw 2", fname)

			plotter.Close()

			err = exec.Command("sync").Run()

			path, err := exec.LookPath("convert")
			args := []string{"convert"}
			args = append(args, "-scale")
			args = append(args, "100x50")
			args = append(args, fmt.Sprintf("%s.png", fname))
			args = append(args, fname+"-small.png")

			if config.Verbose {
				fmt.Fprintf(os.Stderr, "Args:\n")
				for _, arg := range args {
					fmt.Fprintf(os.Stderr, "\t%s\n", arg)
				}
			}

			var procAttr os.ProcAttr
			procAttr.Files = []*os.File{nil, os.Stdout, os.Stderr}

			process, err := os.StartProcess(path, args, &procAttr)
			if err != nil {
				fmt.Fprintf(os.Stderr, "Start process failed %v\n", err)
			}
			_, err = process.Wait()
			if err != nil {
				fmt.Fprintf(os.Stderr, "Wait failed: %v\n", err)
			}

		}
	}


} // main
