/*
 * Copyright (c) 1996 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. Please refer to the file "LICENSE"
 * for further important copyright and licensing information.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 * THIS SOFTWARE IS NOT DESIGNED OR INTENDED FOR USE OR RESALE AS ON-LINE
 * CONTROL EQUIPMENT IN HAZARDOUS ENVIRONMENTS REQUIRING FAIL-SAFE
 * PERFORMANCE, SUCH AS IN THE OPERATION OF NUCLEAR FACILITIES, AIRCRAFT
 * NAVIGATION OR COMMUNICATION SYSTEMS, AIR TRAFFIC CONTROL, DIRECT LIFE
 * SUPPORT MACHINES, OR WEAPONS SYSTEMS, IN WHICH THE FAILURE OF THE
 * SOFTWARE COULD LEAD DIRECTLY TO DEATH, PERSONAL INJURY, OR SEVERE
 * PHYSICAL OR ENVIRONMENTAL DAMAGE ("HIGH RISK ACTIVITIES").  SUN
 * SPECIFICALLY DISCLAIMS ANY EXPRESS OR IMPLIED WARRANTY OF FITNESS FOR
 * HIGH RISK ACTIVITIES.
 */

package java.sql;

/**
 * <P>The java.sql.Numeric class is an arbitrary precision and scale
 * number class that can be used to represent SQL fixed point NUMERIC
 * and DECIMAL values.
 *
 * <P>Numeric objects are composed of four properties: <UL>
 * <LI>Precision : The total number of digits comprising the number,
 * including all the digits to the right and left of the decimal
 * point.
 * <LI>Scale: The number of digits to the right of the decimal point.
 * <LI>Sign:  Positive or negative.
 * <LI>Value: A vector of digits of size <I>precision</I>.
 * </UL>
 *
 * The theoretical limit to the precision and scale values is 0x7fffffff.
 * Numeric objects are immutable.  
 */

public final class Numeric extends java.lang.Number {

    /**
     * Sets the value of the roundingValue which is used in rounding
     * operations.  The roundingValue is a digit between 0 and 9 that
     * determines rounding behavior.  When the scale of a Numeric is
     * reduced either by request or in a calculation, rounding is
     * performed if the value of the scale + 1 digit to the right of
     * the decimal point is greater than the rounding index.
     *
     * <P>For example, if the number is 1.995, and the scale is set to
     * 2 and the roundingValue is 4, then the number would be rounded
     * to 2.00.  The default roundingValue is 4. A
     * roundingValue of 9 cancels rounding.
     *
     * @param val The value between 0 and 9 of the desired roundingValue.
     */
    public static void setRoundingValue(int val) {
	if (val < 0 || val > 9) {
	    throw new IllegalArgumentException("Attempt to set invalid RoundingValue");
	}
	roundingValue = val;
    }

    /**
     * Returns the value of the roundingValue which is used in
     * rounding operations.  The roundingValue is a digit between 0
     * and 9 that determines rounding behavior.  When the scale of a
     * Numeric is reduced either by request or in a calculation,
     * rounding is performed if the value of the scale + 1 digit to
     * the right of the decimal point is greater than the rounding
     * index.
     *
     * <P>For example, if the number is 1.995, and the scale is set to
     * 2 and the RoundingValue is 4, then the number would be rounded
     * to 2.00.  The default roundingValue is 4. A
     * roundingValue of 9 cancels rounding.
     *
     * @return rounding value
     */
    public static int getRoundingValue() {
	return roundingValue;
    }

    /**
     * Creates a numeric from a string. The string may contain a sign
     * and a decimal point, e.g "-55.12".  Spaces and other
     * non-numeric characters are ignored.
     *
     * @param s the string used to initialize the Numeric.
     */

    public Numeric(String s) {
	init(s,-1);
    }

    /**
     * Construct a Numeric given a String and an integer scale.  The
     * scale represents the desired number of places to the right of
     * the decimal point.
     *
     * <P>The String may contain a sign and a decimal point,
     * e.g. "-55.12".  The scale must be an integer with value greater
     * than or equal to zero.  If the scale differs from the implicit
     * decimal point given in the String the value is adjusted to the
     * given scale.  This may involve rounding.
     * 
     * <P>For example, Numeric("99.995",2) would result in a numeric
     * with a scale of 2 and a value of "100.00".  If the String
     * contains no decimal point, then the scale is determined solely
     * by the given integer scale. No implicit decimal point is
     * assumed.  For example, the constructor Numeric("123",2) creates
     * a Numeric with a scale of 2, a precision of 5, and a value of
     * "123.00".  
     *
     * <P><B>Note:</B> Rounding is controlled by the roundingIndex
     * function.
     *
     * @param  s    the string used to initialize the Numeric.
     * @param scale the value greater than or equal to zero
     * representing the desired number of digits to the right of the
     * decimal point.  
     */
    public Numeric(String s,int scale) {
	verifyScale(scale);
	init(s,scale);
    }

    /**
     * Construct a Numeric from an integer given an integer value and
     * an integer scale.  The scale is set to the value specified. No
     * implicit decimal point is assumed, i.e., if the double value is
     * "1234" and the scale 2, the resulting numeric will be "1234.00.
     *
     * @param  x    The double value used to initialize the new Numeric.
     * @param scale The desired number of digits after the decimal point.  
     */
    public Numeric(int x,int scale) {
	verifyScale(scale);
	Integer i = new Integer(x);
	init(i.toString(),scale);
    }

    /**
     * Construct a Numeric from a double given a double value and an
     * integer scale.  The scale is set to the value specified. No
     * implicit decimal point is assumed, i.e., if the double value is
     * "1234" and the scale 2, the resulting numeric will be "1234.00.
     * If the double value is "1234.5" and the scale 2 the value will
     * be "1234.50".  Rounding may occur during this conversion.
     *
     * @param  x    The double value used to initialize the new Numeric.
     * @param scale The desired number of digits after the decimal point.  
     */
    public Numeric(double x, int scale) {
	verifyScale(scale);
	Double d = new Double(x);
	init(d.toString(),scale);
    }

    /**
     * Construct a Numeric from another Numeric.  The precision,
     * scale, and value of the new Numeric are copied from the Numeric
     * given in the argument.
     *
     * @param x The Numeric used to initialize the new Numeric.  
     */
    public Numeric(Numeric x) {
	ourScale = x.ourScale;
	ourPrecision = x.ourPrecision;
	val = new byte[x.val.length];
	for (int i = 0;i < val.length ;i++ ) {
	    val[i] = x.val[i];
	}
	ourSign = x.ourSign;
    }

    /**
     * Given an existing Numeric and an integer scale value, construct
     * a new Numeric with the scale adjusted accordingly.  The
     * precision, scale, and value of the new Numeric are copied from
     * the argument Numeric. The scale is then adjusted to the scale
     * given as a parameter. This may result in rounding of the value,
     * as well as a change to the precision.
     *
     * @param  x		The Numeric to copy.
     * @param scale An integer representing the desired * scale of the
     * new Numeric.  
     */
    public Numeric(Numeric x, int scale) {
	verifyScale(scale);
	ourScale = x.ourScale;
	ourPrecision = x.ourPrecision;
	val = new byte[x.val.length];
	for (int i = 0;i < val.length ;i++ ) {
		val[i] = x.val[i];
	}
	ourSign = x.ourSign;
	if (ourScale != scale) {
		_setScale(scale);
	}
    }

    /**
     * The following methods convert the Numeric value to an
     * appropriate Java built-in type.  Note that these conversions
     * may result in a loss of precision.
     * 
     */

    /**
     * Convert the Numeric value to the Java built-in type of int.
     * This conversion may result in a loss of precision.  Digits
     * after the decimal point are dropped.
     *
     * @return An integer representation of the Numeric value.  
     */
    public int intValue() {
	return ((int)doubleValue());
    }

    /**
     * Convert the Numeric value to the Java built-in type of long.
     * This conversion may result in a loss of precision.  Digits
     * after the decimal point are dropped.
     *
     * @return A long integer representation of the Numeric value.  
     */
    public long longValue() {
	return ((long) doubleValue());
    }
    
    /**
     * Convert the Numeric value to the Java built-in type of float.
     * This conversion may result in a loss of precision.
     *
     * @return A float representation of the Numeric value.  
     */
    public float floatValue() {
	return ((float) doubleValue());
    }

    /**
     * Convert the Numeric value to the Java built-in type of double.
     * This conversion may result in a loss of precision.
     *
     * @return A double representation of the Numeric value.  
     */
    public  double doubleValue() {
	double d=0;
	int i = 0;
	while (i < ourPrecision) {
	    d = d * 10 + (double) val[i++];
	}
	if (ourScale != 0) {
	    double  s = 1;
	    for (int j = 0;j < ourScale ; j++) {
		s *= 10;
	    }
	    d = d/s;
	}
    if(ourSign == 0)
	  return (-d);
	else
	  return d;
    }


    /**
     * Convert the Numeric value to the Java built-in type of String.
     * Negative numbers are represented by a leading "-". No sign is
     * added for positive numbers.
     *
     * @return A String representation of the Numeric value.  
     */
    public  String toString() {
	String x = "";
	byte z = (byte) '0';
	if (ourSign == 0) {
	    x = "-";
        }
	int decpl = ourPrecision - ourScale;
	for (int i=0; i < decpl; i++ ) {
	    x += (char) (z + val[i]);
	}
	if (ourScale > 0) {
	    x += ".";
	    for (int i = decpl; i < ourPrecision; i++ ) {
		x += (char) (z + val[i]);
	    }
	}
	return x;
    }

    /**
     * Return the number of digits to the right of the decimal point.
     *
     *
     * @return An integer value representing the number of decimal
     * places to the right of the decimal point.  
     */
    public int getScale() {
	return (ourScale);
    }

    /**
     * Return the value multiplied by 10**scale. Precision may be
     * lost.  Thus, if a currency value was being kept with scale "2"
     * as "5.04", the getScaled function would return the long integer
     * "504".
     *
     * @return The scaled value as a long.  
     */
    public long getScaled() {
	if (ourScale == 0) {
	    return (longValue());
	}
	if (ourScale < 9) {
	    long s= 1;
	    for (int i=0; i < ourScale; i++) {
		s *= 10;
	    }
	    return ((multiply(new Numeric(s,0))).longValue());
	}
	Numeric ten = new Numeric(10,0);
	Numeric s = new Numeric(1,0);
	for (int i = 0; i < ourScale; i++) {
	    s = s.multiply(ten);
	}
	return ((multiply(s)).longValue());
    }

    /**
     * Return a numeric value created by dividing the argument by
     * 10**scale.  Thus, createFromScaled(504,2) would create the
     * value "5.04".
     *
     * @param  scaled The scaled value as a long.
     * @param  s      The desired scale value as an integer between 0 and 9
     * @return A new numeric.  
     */
    public static Numeric createFromScaled(long scaled, int s) {
	Long x = new Long(scaled);
	Numeric n = new Numeric(x.toString());
	if (s == 0) {
	    return (n);
	} else {
	    Numeric ten = new Numeric(10,0);
	    Numeric ftor = new Numeric(1,0);
	    for (int i = 0; i < s ; i++) {
		ftor = ftor.multiply(ten);
	    }
	    n._setScale(s);
	    return (n.divide(ftor));
	}
    }
    /**
     * Returns the arithmetic sum of the Numeric and the argument.
     * The scale of the sum is determined by this object not by the
     * argument.
     *
     * @param  n  The Numeric to add.
     * @return The sum as a Numeric.  
     */
    public Numeric add(Numeric n) {
	Numeric x= new Numeric(n,ourScale);
	Numeric result= new Numeric(this);

	if (result.ourSign == x.ourSign) {
	    result = (result.plus(x));
	    result.ourSign = ourSign;
	    return (result);
	}

	if (result.ourSign == 1) {
	    int cmp = result.compareAbs(x);
	    if (cmp < 0) {
		result = x.sub(this);
		result.ourSign = x.ourSign;
		return (result);
	    } else {
		if (cmp > 0) {
		    result = result.sub(x);
		    return (result);
		} else {
		    result.zero();
		    return (result);

		}
	    }
	}
	int cmp = result.compareAbs(x);
	if (cmp < 0) {
	    result=x.sub(this);
	    result.ourSign = x.ourSign;
	    return (result);
	} else {
	    if (cmp > 0) {
		result=result.sub(x);
		return (result);
	    }
	}
	result.zero();
	return (result);
    }

    /**
     * Returns the arithmetic difference between the Numeric and the
     * argument.  The scale of the sum is determined by this object
     * not by the argument.
     *
     * @param  n The Numeric to subtract.
     * @return The difference as a Numeric.  
     */
    public Numeric subtract(Numeric n) {
	Numeric x = new Numeric(n,ourScale);
	Numeric result = new Numeric(this);


	if (result.ourSign == 1 && x.ourSign == 1) {
	    int c = result.compareAbs(x);
	    if (c == 0) {
	        result.zero();
		return (result);

	    }
	    if (c == 1) {
		return (result.sub(x));
            }
	    result = x.sub(this);
	    result.ourSign = 0;
	    return (result);
	}

	if (result.ourSign == 1 && x.ourSign == 0) {
	    return (result.plus(x));
	}

	if (result.ourSign == 0 && x.ourSign == 1) {
	    result = result.plus(x);
	    result.ourSign = 0;
	    return (result);
	}

	int c = result.compareAbs(x);
	if (c == 0) {
	    result.zero();
	    return (result);
	}
	if (c == -1) {
	    result = (x.sub(result));
	    result.ourSign = 1;
	    return (result);
	}

	result = result.sub(x);
	result.ourSign = 0;
	return (result);
    }

    /**
     * Returns the arithmetic product of the object Numeric and the
     * argument.  The scale of the product is determined by this object
     * not by the argument.
     *
     * @param  x The multiplier as a Numeric.
     * @return The product as a Numeric.  
     */
    public Numeric multiply(Numeric x) {
	int saveScale;
	Numeric prod = new Numeric(this); //product
	Numeric m; //multiplicand
	Numeric mx; //multiplier

	saveScale = ourScale;
	prod.ourScale += x.ourScale;
	prod.ourPrecision += x.ourPrecision + 1;
	if (prod.ourPrecision > MAXNUMERICLEN || prod.ourPrecision < 1) {
	    throw new ArithmeticException(
		"Overflow in Numeric multiply - precison (" + 
	        prod.ourPrecision + ") exceeds maximum precision of " + 
		MAXNUMERICLEN);
        }
	prod.val = new byte[x.val.length + val.length + 2];
	prod.zero();
	prod.ourSign = (byte)(ourSign == x.ourSign? 1: 0);

	if (ourPrecision > x.ourPrecision) {
	    //use the smallest number as the multiplier
	    m = new Numeric(this);
	    saveScale = ourScale;
	    mx = new Numeric(x);
	} else {
	    m = new Numeric(x);
	    mx = new Numeric(this);
	}

	int carry = 0;
	int im = m.ourPrecision - 1; //index into multiplicand
	int imx = mx.ourPrecision - 1; //index into multiplier
	int iprod = prod.ourPrecision - 1;
	int mx_len = mx.ourPrecision;
	int m_len = m.ourPrecision;
	while (mx_len > 0) {
	    int j = im;
	    int p = iprod;
	    while (j >= 0) {
		if (mx.val[imx] == 0) {
		    break;
                }
		int v = mtbl[mx.val[imx] + m.val[j] * 10];
		prod.val[p] += carry + (v & 0x00FF);
		carry = v >> 8;
		if (prod.val[p] > 9) {
		    carry += prod.val[p]/10;
		    prod.val[p] %= 10;
		}
		j--;
		p--;
	    }
	    if (carry > 0) {
		prod.val[p] += (byte) carry;
		carry = 0;
	    }
	    mx_len--;
	    imx--;
	    iprod--;
	}
	prod._setScale(saveScale); //set the scale to this scale
	prod.dropLeadingZeroes();
	return (prod);
    }

    /**
     * Returns the arithmentic quotient calculated by dividing the
     * Numeric by the argument.  The scale of the quotient is
     * determined by this object not by the argument.
     *
     * @param  x The divisor as a Numeric.
     * @return The quotient as a Numeric.  
     */
    public Numeric divide(Numeric x) {
	Numeric dvdnd =  new Numeric(this); //dividend
	Numeric div = new Numeric(x);       //divisor

	int saveScale = dvdnd.ourScale;
	div.dropLeadingZeroes();
	if (div.compareAbs(new Numeric("0.0",(int)div.ourScale)) == 0) {
	    throw new ArithmeticException("Division by zero");
	}
	// setup the dividend
	dvdnd.dropLeadingZeroes();
	//get the first two digits of the divisor
	int d1 = div.val[0];
	// normalize
	if (d1 < 5) {
	    int d = 10/(d1 + 1);
	    if (d > 1) {
		Numeric temp = new Numeric(d,0);
		dvdnd = dvdnd.multiply(temp);	 //optimize this later
		div = div.multiply(temp);
		d1 = div.val[0];

	    }
	}
	int d2 = div.val[1];
	// add a leading zero
	byte tval[] = new byte[dvdnd.ourPrecision + div.ourScale + 2]; //move decimal pt and add leading zero
	// and one added scale precision

	tval[0] = 0; //add a leading zero
	int i=1,j=0;
	for (; j < dvdnd.ourPrecision; j++,i++) {
	    tval[i] = dvdnd.val[j];
	}
	while (i < tval.length) {
	    tval[i++] = 0;
	}
	dvdnd.ourPrecision= tval.length;
	dvdnd.val = tval;
	dvdnd.ourScale++;
	//setup the quotient
	Numeric quot = new Numeric(dvdnd);
	quot.zero();
	quot.ourSign = (byte) (dvdnd.ourSign == div.ourSign ? 1 : 0);
	//initialize loop
	j = 0;
	int q = 0;
	while (j < dvdnd.ourPrecision - div.ourPrecision) {
	    // estimate q
	    int dd0,dd1,dd2;   //first three digits of dividend relative to j
	    dd0 = dvdnd.val[j];
	    dd1 = dvdnd.val[j+1];
	    dd2 = dvdnd.val[j+2];
	    if (dd1 == d1) {
		q=9;
	    } else {
		q = (dd0*10 + dd1)/d1;
	    }
	    while ((d2*q) > ((dd0*10 + dd1 - q*d1)* 10 + dd2)) {
		q--;
	    }
	    if (q==0) {
		quot.val[j + div.ourPrecision] = (byte) q;
		j++;
		continue;
	    }
	    int k = j + div.ourPrecision;
	    int carry = 0;
	    int brw = 0;    //borrow amount
	    for (int dk = div.ourPrecision - 1;dk >= 0 ;dk--,k--) {

		int x1 = mtbl[div.val[dk] + (q * 10)];
		int x2 = carry + (x1 & 0x00FF);
		carry = x1 >> 8;
		if (x2 > 9) {
		    carry += x2/10;
		    x2 %= 10;
		}
		x1 = (int) dvdnd.val[k] - ( x2 + brw);
		if (x1 < 0) {
		    brw = 1;
		    x1 += 10;
		} else {
		    brw = 0;
		}
		dvdnd.val[k] = (byte) x1;
	    }
	    if (brw > 0 ) { //the guess may be too big! back it off by 1
		int x1= -1;
		if (k >=0) {
		    x1 = dvdnd.val[k] - (carry + brw);
		}
		if (x1 < 0) {
		    q--;
		    k = j + div.ourPrecision;
		    carry = 0;
		    for (int dk = div.ourPrecision - 1;dk >= 0 ;dk--,k--) {
			dvdnd.val[k] = (byte) (dvdnd.val[k] + div.val[dk] + carry);
			if (dvdnd.val[k] >= 10) {
			    carry = 1;
			    dvdnd.val[k] -= 10;
			} else {
			    carry = 0;
			}
		    }

		}
	    }
	    quot.val[j + div.ourPrecision] = (byte) q;
	    j++; //next dividend
	}
	quot._setScale(saveScale);
	return (quot);
    }

    /**
     * Returns true if the arithmetic value of the Numeric equals the
     * argument.
     *
     * @param  obj The object to compare.
     * @return The boolean result of the comparison.  
     */
    public  boolean equals(Object obj) {
	Numeric x = (Numeric) obj;
	if (compare(x) == 0) {
	    return (true);
	}
	return (false);

    }

    /**
     * Returns true if the arithmetic value of the Numeric is less
     * than the argument.
     *
     * @param  x The object to compare.
     * @return The boolean result of the comparison.  
     */
    public boolean lessThan(Numeric x) {
	if (compare(x) == -1) {
            return (true);
	}
	return (false);
    }

    /**
     * Returns true if the arithmetic value of the Numeric is less
     * than or equals the argument.
     *
     * @param  x The object to compare.
     * @return The boolean result of the comparison.  
     */
    public  boolean lessThanOrEquals(Numeric x) {
 	if (compare(x) != 1) {
            return (true);
	}
 	return (false);
    }

    /**
     * Returns true if the arithmetic value of the Numeric is greater
     * than the argument.
     *
     * @param  x The object to compare.
     * @return The boolean result of the comparison.  
     */
    public  boolean greaterThan(Numeric x) {
	if (compare(x) == 1) {
	    return (true);
	}
	return (false);
    }

    /**
     * Returns true if the arithmetic value of the Numeric is greater
     * than or equals the argument.
     *
     * @param  x The object to compare.
     * @return The boolean result of the comparison.  
     */
    public  boolean greaterThanOrEquals(Numeric x) {
	if (compare(x) != -1) {
 	    return (true);
	}
	return (false);
    }

    /**
     * Returns an integer hashcode for the Numeric.  Note that the
     * code returned for 03.00 will equal the code for 3.0.
     *
     * @return The hashcode as an integer.  
     */
    public int hashCode() {
	Numeric temp = new Numeric(this);
	temp.dropLeadingZeroes();
	//now drop trailing zeroes
	for (int i = temp.ourPrecision - 1;i > 0; i-- ) {
	    if (temp.val[i] != 0) {
		break;
	    }
	    temp.ourPrecision--;
	    temp.ourScale--;
	}
	if (temp.ourPrecision == 0) {
	    temp.ourPrecision = 1;
	    temp.ourScale = 0;
	    temp.val[0] = 0;
	}
	return (temp.toString().hashCode());

    }

    /**
     * Returns a numeric copied from the current object with the scale
     * adjusted as specified by the argument.  Sets the number of the
     * digits to the right of the decimal point.  The precision and
     * value of the Numeric will be adjusted accordingly.  Note that
     * changing the scale to a smaller value may result in rounding
     * the value.
     *
     * @param  scale the character to be converted
     * @return A Numeric with the adjusted scale.
     * @see Numeric#roundingValue 
     */
    public Numeric setScale(int scale) {
	verifyScale(scale);
	Numeric x = new Numeric(this,scale);
	return (x);
    }

   //------------------------------------------Implementation---------------------------
    public  void debug() {

  	System.out.println("Precision = " + ourPrecision + ",Scale=" + ourScale
	                   + ",Sign = " + ourSign + ",Value = ->" + toString() + "<-");
    }
    static int mtbl[] =  {											//multiplication table
	0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
	0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,0x0008,0x0009,
	0x0000,0x0002,0x0004,0x0006,0x0008,0x0100,0x0102,0x0104,0x0106,0x0108,
	0x0000,0x0003,0x0006,0x0009,0x0102,0x0105,0x0108,0x0201,0x0204,0x0207,
	0x0000,0x0004,0x0008,0x0102,0x0106,0x0200,0x0204,0x0208,0x0302,0x0304,
	0x0000,0x0005,0x0100,0x0105,0x0200,0x0205,0x0300,0x0305,0x0400,0x0405,
	0x0000,0x0006,0x0102,0x0108,0x0204,0x0300,0x0306,0x0402,0x0408,0x0504,
	0x0000,0x0007,0x0104,0x0201,0x0208,0x0305,0x0402,0x0409,0x0506,0x0603,
	0x0000,0x0008,0x0106,0x0204,0x0302,0x0400,0x0408,0x0506,0x0604,0x0702,
	0x0000,0x0009,0x0108,0x0207,0x0306,0x0405,0x0504,0x0603,0x0702,0x0801
    };
    /**
     * The maximum precision allowed. Set by default to 255.
     * 
     */
    private static final int MAXNUMERICLEN = 0x7fffffff;	 //Maximum numeric precision.

    /**
     * The maximum scale supported. Set by default to 0x7ffffffff.
     *
     */
    private static final int MAXSCALE = 0x7fffffff;          //Maximum scale.

    /**
     * The roundingValue is a digit between 0 and 9 that determines
     * rounding behavior.  When the scale of a numeric is reduced
     * either by request or in a calculation, rounding is performed if
     * the value of the scale + 1 digit to the right of the decimal
     * point is greater than the rounding index.
     *
     * <P>For example, if the number is 1.995, and the scale is set to
     * 2 and the roundingValue is 4, then the number would be rounded
     * to 2.00.  The default roundingValue is 4. A roundingValue of 9
     * cancels rounding.  
     */
    private static int roundingValue = 4;

    private int ourPrecision;   // number of digits in the number, 
				// left and right of the decimal point
    private int ourScale;	// number of digits right of the decimal point
    private byte ourSign;       // 1 if positive, 0 if negative
    private byte val[];		// byte array holding the digit values
    
    /**
     * Throws IllegalArgument exception is the scale is negative or
     * greater then MAXSCALE (255).
     * 
     */
    private void verifyScale(int scale) {

	if (scale < 0) {
	    throw new IllegalArgumentException("Scale is negative");
	}
	if (scale > MAXSCALE) {
	    throw new IllegalArgumentException("Scale greater than maximum scale");
	}
    }

    /**
     * Private function used for arithmetic comparisons.
     *
     * @param  B  A numeric to compare.
     * @return	  1 if this > B,0 if this = B,-1 if this < B.
     */
    private int compare(Numeric B) {
	if (ourSign == 1 && B.ourSign == 1) {
	    return (compareAbs(B));
	}
	if (ourSign == 0 && B.ourSign == 1) {
	    return (-1);
	}
	if (ourSign == 1 && B.ourSign == 0) {
	    return (1);
	}
	int r = compareAbs(B);
	if (r != 0) {
	    r *= -1;
	}
	return (r);
    }

    /**
     * Private function used for arithmetic comparisons.  Performs an
     * absolute comparison.
     *
     * @param  A A numeric to compare.
     * @return 1 if this > A,0 if this = A,-1 if this < A.  
     */
    private int compareAbs(Numeric A) {
	Numeric B = new Numeric(A,ourScale);

	//look for the first significant digit
	int sigA,sigAA,sigB,sigBB;
	for (sigB = 0;sigB < B.ourPrecision ; sigB++ ) {
	    if (B.val[sigB] != 0) {
		break;
	    }
	}
	for (sigA = 0;sigA < ourPrecision ; sigA++ ) {
	    if (val[sigA] != 0) {
		break;
	    }
	}
	sigAA = ourPrecision - sigA;
	sigBB = B.ourPrecision - sigB;
	if (sigAA > sigBB) {
	    return (1);
	} else {
	    if (sigBB > sigAA) {
		return (-1);
	    }
	}
	if (sigAA == 0 && sigBB == 0) {
	    return (0);
	}
	while (sigA < ourPrecision )  {
	    if (val[sigA] > B.val[sigB]) {
		return (1);
	    }
	    if (val[sigA] < B.val[sigB]) {
		return (-1);
	    }
	    sigA++;
	    sigB++;
	}
	return (0);
    }

    /**
     * Private function used to initialize a numeric using a string.
     *
     * @param  s	A string containing the intended value of the numeric.
     * @param  scale    The number of digits to the right of the decimal point.
     *                  Should be -1 if the scale is to be determined by the contents
     *                  of the argument string s.
     */
    private void init(String s, int scale) {
	boolean scaleGiven = true;
	ourScale = 0;

	if (scale < 0) {
	    scaleGiven = false;
	}
	int inScale = 0; //Scale computed from decimal point in s


	ourSign = 1; 		//Default to positive
	ourPrecision = 0;
	if (scaleGiven) {
	    val = new byte[s.length() + scale];
	} else {
	    val = new byte[s.length()];
	}
	int j = 0;
	for (int i = 0;i < s.length() ;i++ ) {
	    char c = s.charAt(i);
	    if (c >= '0' && c <= '9') {
		val[j++] = (byte) (c - '0');
		ourPrecision++;
		if (inScale != 0) {
		    inScale++;
		}
		continue;
	    }
	    if (c == '-') {
		ourSign = 0;
		continue;
	    }
	    if (c == '.') {
		inScale = 1;
	    }
	}
	if (inScale != 0) {		   
	    //did we find a decimal point in the string
	    inScale--;
	    ourScale = (byte) inScale;	   
	    //if so, use it
	    if (scaleGiven) {
		if (ourScale > scale)	{  
		    //the number of decimal places in string exceeds request
		    _setScale(scale);
		}

		while (ourScale < scale)	{
	            //number of decimal places is less than request
		    val[j++] = (byte) 0;
		    ourPrecision++;
		    ourScale++;
		}
	    }
	} else { 
	    // there was no decimal point in the string
	    ourScale = 0;
	    if (scaleGiven) {
		_setScale(scale);
	    }
	}
    }

    /**
     * Private function used in addition. Performs unsigned addition.
     *
     * @param  x  The numeric to add.
     * @return	  The sum as a Numeric.
     */
    private Numeric plus(Numeric x) {
	x._setScale(ourScale);
	Numeric result = new Numeric(this);
	result.ourPrecision = (ourPrecision + 1);
	result.val = new byte[result.ourPrecision];

	int sigA,sigAA,sigB,sigBB;
	for (sigB = 0;sigB < x.ourPrecision ; sigB++ ) {
	    if (x.val[sigB] != 0) {
		break;
	    }
	}
	for (sigA = 0;sigA < ourPrecision ; sigA++ ) {
	    if (val[sigA] != 0) {
		break;
	    }
	}
	int a = ourPrecision - 1;
	int b = x.ourPrecision - 1;
	int i = result.ourPrecision - 1;
	int carry=0,sum;

	while (b >= sigB) {
	    //	result.val[i] = (byte) addtbl[(val[a] * 11) + (x.val[b] + carry)];
	    sum = val[a]  + x.val[b] + carry;
	    if (sum >= 10) {
		carry = 1;
		sum -= 10;
	    } else {
		carry = 0;
	    }
	    result.val[i] = (byte) sum;
	    a--;b--;i--;
	}
	while (a >= sigA) {
	    sum = val[a] + carry;
	    if (sum >= 10) {
		carry = 1;
		sum -= 10;
	    } else {
		carry = 0;
	    }
	    result.val[i] = (byte) sum;
	    i--; a--;
	}
	result.val[i--] = (byte) carry;
	while (i >= 0) {
	    result.val[i] = 0;
	    i--;
	}
	result.dropLeadingZeroes();
	return (result);
    }

    /**
     * Private function used in subtraction. Performs unsigned
     * subtraction.
     *
     * @param  n  The numeric to subtract.
     * @return	  The difference as a Numeric.
     * @see 
     */
    private Numeric sub(Numeric n) {
	Numeric x = new Numeric(n);
	x._setScale(ourScale);
	Numeric result = new Numeric(this);
	if (x.ourPrecision > ourPrecision) {
	    result.ourPrecision = x.ourPrecision;
	    result.val = new byte[x.val.length];
	}
	int sigA,sigAA,sigB,sigBB;
	for (sigB = 0;sigB < x.ourPrecision ; sigB++ ) {
	    if (x.val[sigB] != 0) {
		break;
	    }
	}
	for (sigA = 0;sigA < ourPrecision ; sigA++ ) {
	    if (val[sigA] != 0) {
		break;
	    }
	}
	int a = ourPrecision - 1;
	int b = x.ourPrecision - 1;
	int i = result.ourPrecision - 1;
	int carry=0,diff;

	while (b >= sigB) {
	    diff = val[a] - (x.val[b] + carry);
	    if (diff < 0) {
		carry = 1;
		diff  += 10;
	    } else {
		carry = 0;
	    }
	    result.val[i] = (byte) diff ;
	    a--;b--;i--;
	}
	while (a >= sigA) {
	    diff = val[a] -  carry;
	    if (diff < 0) {
		carry = 1;
		diff += 10;
	    } else {
		carry = 0;
	    }
	    result.val[i] = (byte) diff;
	    i--; a--;
	}
	if (carry != 0) {
	    result.val[i--] = (byte) carry;
	}
	while (i >= 0) {
	    result.val[i] = 0;
	    i--;
	}
	return (result);
    }

    /**
     * Set the value of the Numeric to zeroes.  The precision and
     * scale are retained.  
     */
    private void zero() {
	ourSign = 1;
	for (int i = 0; i < ourPrecision; i++) {
	    val[i] = 0;
	}
    }

    /**
     * Remove leading zeroes from the value of the Numeric. Zeroes to
     * the right of the decimal point are retained.
     *
     * <P>The precision is changed accordingly. One zero is retained
     * if all digits are zero and the scale is zero.  
     */
    private void dropLeadingZeroes() {
	int i;
	for (i = 0; i < (ourPrecision - ourScale); i++) {
	    if (val[i] != 0) {
		break;
	    }
	}
	if (i < (ourPrecision - ourScale)) {
	    byte nval[] = new byte[ourPrecision - i];
	    for (int j = 0;i < ourPrecision ; i++,j++ ) {
		nval[j] = val[i];
	    }
	    val = nval;
	    ourPrecision = nval.length;
	}

    }
    
    /**
     * Set the number of the digits to the right of the decimal point.
     * The precision and value of the Numeric will be adjusted
     * accordingly.  Note that changing the scale to a smaller value
     * may result in rounding the value.
     *
     * @param  scale the character to be converted
     * @see Numeric#roundingValue 
     */
    private void _setScale(int scale) {
	verifyScale(scale);
	if (scale == ourScale) {
	    return;
	}
	if (scale < ourScale) {
	    int savPrecision = ourPrecision; //track rounding
	    while (scale < ourScale && ourScale > 0) {
		ourScale--;
		ourPrecision--;
	    }
	    if (ourPrecision <= 0) {
		ourPrecision = 1;
		val[0] = 0;
	    }
	    int r = 0; //scale + 1 digit
	    if (savPrecision > ourPrecision && savPrecision > 1) {
		r = val[ourPrecision];
	    }
	    if (r > roundingValue) {	 
		// round up if desired
		int carry = 1;
		for (int k=ourPrecision - 1; k >= 0 ; k--) {
		    val[k] = (byte) (val[k]  + carry);
		    if (val[k] >= 10) {
			carry = 1;
			val[k] -= 10;
		    } else {
			carry = 0;
		    }
		}
		if (carry > 0) {	//overflow
		    byte temp[] = new byte[ourPrecision + 1];
		    for (int ii = ourPrecision - 1,jj = temp.length - 1;ii >= 0;ii--,jj--) {
			temp[jj] = val[ii];
		    }
		    temp[0] = (byte) carry;
		    val = temp;
		    ourPrecision =  temp.length;
		}
	    }
	}

	int i = ourPrecision;
	if (val.length < (i + scale - ourScale + 1)) {
	    byte temp[] = new byte[val.length + (scale - ourScale + 1)];
	    for (int ii = 0;ii < val.length;ii++) {
		temp[ii] = val[ii];
	    }
	    val = temp;
	}
	while (scale > ourScale) {
	    val[i++] = (byte) 0;
	    ourScale++;
	}
	ourPrecision =  i;
    }
}
