/*
 * Copyright (c) 1996 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. Please refer to the file "LICENSE"
 * for further important copyright and licensing information.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 * THIS SOFTWARE IS NOT DESIGNED OR INTENDED FOR USE OR RESALE AS ON-LINE
 * CONTROL EQUIPMENT IN HAZARDOUS ENVIRONMENTS REQUIRING FAIL-SAFE
 * PERFORMANCE, SUCH AS IN THE OPERATION OF NUCLEAR FACILITIES, AIRCRAFT
 * NAVIGATION OR COMMUNICATION SYSTEMS, AIR TRAFFIC CONTROL, DIRECT LIFE
 * SUPPORT MACHINES, OR WEAPONS SYSTEMS, IN WHICH THE FAILURE OF THE
 * SOFTWARE COULD LEAD DIRECTLY TO DEATH, PERSONAL INJURY, OR SEVERE
 * PHYSICAL OR ENVIRONMENTAL DAMAGE ("HIGH RISK ACTIVITIES").  SUN
 * SPECIFICALLY DISCLAIMS ANY EXPRESS OR IMPLIED WARRANTY OF FITNESS FOR
 * HIGH RISK ACTIVITIES.
 */

package java.sql;

/**
 * <P>A Connection represents a session with a specific
 * database. Within the context of a Connection, SQL statements are
 * executed and results are returned.
 *
 * <P>A Connection's database is able to provide information
 * describing its tables, its supported SQL grammar, its stored
 * procedures, the capabilities of this connection, etc. This
 * information is obtained with the getMetaData method.
 *
 * <P><B>Note:</B> By default the Connection automatically commits
 * changes after executing each statement. If auto commit has been
 * disabled an explicit commit must be done or database changes will
 * not be saved.
 *
 * @see DriverManager#getConnection
 * @see Statement 
 * @see ResultSet
 * @see DatabaseMetaData
 */
public interface Connection {

    /**
     * SQL statements without parameters are normally
     * executed using Statement objects. If the same SQL statement 
     * is executed many times it is more efficient to use a 
     * PreparedStatement
     *
     * @return a new Statement object 
     */
    Statement createStatement() throws SQLException;

    /**
     * A SQL statement with or without IN parameters can be
     * pre-compiled and stored in a PreparedStatement object. This
     * object can then be used to efficiently execute this statement
     * multiple times.
     *
     * <P><B>Note:</B> This method is optimized for handling
     * parametric SQL statements that benefit from precompilation. If
     * the driver supports precompilation, prepareStatement will send
     * the statement to the database for precompilation. Some drivers
     * may not support precompilation. In this case, the statement may
     * not be sent to the database until the PreparedStatement is
     * executed.  This has no direct affect on users; however, it does
     * affect which method throws certain SQLExceptions.
     *
     * @param sql a SQL statement that may contain one or more '?' IN
     * parameter placeholders
     *
     * @return a new PreparedStatement object containing the
     * pre-compiled statement 
     */
    PreparedStatement prepareStatement(String sql)
	    throws SQLException;

    /**
     * A SQL stored procedure call statement is handled by creating a
     * CallableStatement for it. The CallableStatement provides
     * methods for setting up its IN and OUT parameters, and
     * methods for executing it.
     *
     * <P><B>Note:</B> This method is optimized for handling stored
     * procedure call statements. Some drivers may send the call
     * statement to the database when the prepareCall is done; others
     * may wait until the CallableStatement is executed. This has no
     * direct affect on users; however, it does affect which method
     * throws certain SQLExceptions.
     *
     * @param sql a SQL statement that may contain one or more '?'
     * parameter placeholders
     *
     * @return a new CallableStatement object containing the
     * pre-compiled SQL statement 
     */
    CallableStatement prepareCall(String sql) throws SQLException;
						
    /**
     * A driver may convert the JDBC sql grammar into its system's
     * native SQL grammar prior to sending it; nativeSQL returns the
     * native form of the statement that the driver would have sent.
     *
     * @param sql a SQL statement that may contain one or more '?'
     * parameter placeholders
     *
     * @return the native form of this statement
     */
    String nativeSQL(String sql) throws SQLException;

    /**
     * If a connection is in auto-commit mode, then all its SQL
     * statements will be executed and committed as individual
     * transactions.  Otherwise, its SQL statements are grouped into
     * transactions that are terminated by either commit() or
     * rollback().  By default, new connections are in auto-commit
     * mode.
     *
     * @param autoCommit true enables auto-commit; false disables
     * auto-commit.  
     */
    void setAutoCommit(boolean autoCommit) throws SQLException;

    /**
     * Get the current auto-commit state.
     * @return Current state of auto-commit mode.
     * @see #setAutoCommit 
     */
    boolean getAutoCommit() throws SQLException;

    /**
     * Commit makes all changes made since the previous
     * commit/rollback permanent and releases any database locks
     * currently held by the Connection. 
     *
     * <P><B>Note:</B> By default, a Connection's PreparedStatements,
     * CallableStatements and ResultSets are implicitly closed when it
     * is committed.
     *
     * @see #setAutoClose 
     * @see #setAutoCommit 
     */
    void commit() throws SQLException;

    /**
     * Rollback drops all changes made since the previous
     * commit/rollback and releases any database locks currently held
     * by the Connection.
     *
     * <P><B>Note:</B> By default, a Connection's PreparedStatements,
     * CallableStatements and ResultSets are implicitly closed when it
     * is committed.
     *
     * @see #setAutoClose 
     * @see #setAutoCommit 
     */
    void rollback() throws SQLException;

    /**
     * In some cases, it is desirable to immediately release a
     * Connection's database and JDBC resources instead of waiting for
     * them to be automatically released; the close method provides this
     * immediate release. 
     *
     * <P><B>Note:</B> A Connection is automatically closed when it is
     * garbage collected. Certain fatal errors also result in a closed
     * Connection.
     */
    void close() throws SQLException;;

    /**
     * Check if a Connection is closed
     *
     * @return true if the connection is closed; false if it's still open
     */
    boolean isClosed() throws SQLException;;

    //======================================================================
    // Advanced features:

    /**
     * A Connection's database is able to provide information
     * describing its tables, its supported SQL grammar, its stored
     * procedures, the capabilities of this connection, etc. This
     * information is made available through a DatabaseMetaData
     * object.
     *
     * @return a DatabaseMetaData object for this Connection 
     */
    DatabaseMetaData getMetaData() throws SQLException;;

    /**
     * You can put a connection in read-only mode as a hint to enable 
     * database optimizations.
     *
     * <P><B>Note:</B> setReadOnly cannot be called while in the
     * middle of a transaction.
     *
     * @param readOnly true enables read-only mode; false disables
     * read-only mode.  
     */
    void setReadOnly(boolean readOnly) throws SQLException;

    /**
     * Test if the connection is in read-only mode
     *
     * @return true if connection is read-only
     */
    boolean isReadOnly() throws SQLException;

    /**
     * A sub-space of this Connection's database may be selected by setting a
     * catalog name. If the driver does not support catalogs it will
     * silently ignore this request.
     */
    void setCatalog(String catalog) throws SQLException;

    /**
     * Return the Connection's current catalog name
     *
     * @return the current catalog name or null
     */
    String getCatalog() throws SQLException;

    /**
     * Transactions are not supported. 
     */
    int TRANSACTION_NONE	     = 0;

    /**
     * Dirty reads are done
     */
    int TRANSACTION_READ_UNCOMMITTED = 1;

    /**
     * Only reads on the current row are repeatable 
     */
    int TRANSACTION_READ_COMMITTED   = 2;

    /**
     * Reads on all rows of a result are repeatable
     */
    int TRANSACTION_REPEATABLE_READ  = 4;

    /**
     * Reads on all rows of a transaction are repeatable
     */
    int TRANSACTION_SERIALIZABLE     = 8;

    /**
     * You can call this method to try to change the transaction
     * isolation level on a newly opened connection, using one of the
     * TRANSACTION_* values.  
     *
     * <P><B>Note:</B> setTransactionIsolation cannot be called while
     * in the middle of a transaction.
     *
     * @param level one of the TRANSACTION_* isolation values with the
     * exception of TRANSACTION_NONE; some databases may not support
     * other values
     *
     * @see DatabaseMetaData#supportsTransactionIsolationLevel
     */
    void setTransactionIsolation(int level) throws SQLException;

    /**
     * Get this Connection's current transaction isolation mode
     *
     * @return the current TRANSACTION_* mode value
     */
    int getTransactionIsolation() throws SQLException;

    /**
     * When a Connection is in auto-close mode all its
     * PreparedStatements, CallableStatements, and ResultSets will be
     * closed when a transaction is committed or rolled back.  By
     * default, a new Connection is in auto-close mode.
     *
     * <P> When auto-close is disabled JDBC attempts to keep 
     * all statements and ResultSets open across commits and
     * rollbacks.  However the actual behaviour will vary depending
     * on what the underlying database supports.  Some databases
     * allow these objects to remain open across commits whereas
     * other databases insist on closing them.
     *
     * @param autoClose true enables auto-close, false disables
     * 		auto-close.
     *
     * @see DatabaseMetaData#supportsOpenCursorsAcrossCommit
     * @see DatabaseMetaData#supportsOpenCursorsAcrossRollback
     * @see DatabaseMetaData#supportsOpenStatementsAcrossCommit
     * @see DatabaseMetaData#supportsOpenStatementsAcrossRollback 
     */
    void setAutoClose(boolean autoClose) throws SQLException;
	
    /**
     * Get the current auto-close state.
     * @return Current state of auto-close mode.
     * @see #setAutoClose 
     */
    boolean getAutoClose() throws SQLException;

    /**
     * The first warning reported by calls on this Connection is
     * returned.  
     *
     * <P><B>Note:</B> Subsequent warnings will be chained to this
     * SQLWarning.
     *
     * @return the first SQLWarning or null 
     */
    SQLWarning getWarnings() throws SQLException;

    /**
     * After this call getWarnings returns null until a new warning is
     * reported for this Connection.  
     */
    void clearWarnings() throws SQLException;
}
