/*	Modified by Rukmangathan Balakrishnan 
	for PDFS.

*/

/*
 * fh_cache.h		This module handles the file-handle cache.
 *
 * Authors:	Mark A. Shand, May 1988
 *		Don Becker, <becker@super.org>
 *		Rick Sladkey, <jrs@world.std.com>
 *		Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *
 *		Copyright 1988 Mark A. Shand
 *		This software maybe be used for any purpose provided
 *		the above copyright notice is retained.  It is supplied
 *		as is, with no warranty expressed or implied.
 */

#ifndef FH_CACHE_H
#define FH_CACHE_H

#include <sys/types.h>
#include <sys/stat.h>

#include "pdfs.h"
#include "args.h"

#define FHC_PDFSMOUNTED		004

#define ROOT_UID	0	// super user id

/* Modes for fh_find */
enum {
	FHFIND_FEXISTS = 0,	/* file must exist */
	FHFIND_FCREATE,		/* file will be created */
	FHFIND_FCACHED,		/* fh must be in cache */
};

 // This defines the maximum number of handles pdfsd will cache.
#define	FH_CACHE_LIMIT		2000

/*
 * This defines the maximum number of file PDFSd may keep open for PDFS I/O.
 * It used to be 8...
 */
#define FD_CACHE_LIMIT		(3*FOPEN_MAX/4)

/* The following affect cache expiry.
 * CLOSE_INTERVAL applies to the closing of inactive file descriptors
 * The fd expiry interval is actually quite low because we want to have big
 * files actually go away when they have been deleted behind our back. 
 * We also want to be able to execute programs that have just been copied
 * via PDFS.
 *
 * DISCARD_INTERVAL is the time in seconds PDFSd will cache file handles
 * unless it's being flooded with other requests. This value is possibly
 * still too large, but the original was 2 days.		--okir
 */

#define FLUSH_INTERVAL		5			/* 5 seconds	*/
#define BUSY_RETRY_INTERVAL	2			/* 2 seconds	*/
#define CLOSE_INTERVAL		5			/* 5 seconds	*/
#define DISCARD_INTERVAL	(60*60)			/* 1 hour	*/

/*
 * Hashed search path to this file.
 * path is: hash_path[1] ... hash_path[hash_path[0]]
 *
 * hash_path[hash_path[0]+1] ... hash_path[HP_LEN-1] == 0
 */
typedef struct {
	psi_t		psi;
	unsigned char 	hash_path[HP_LEN];
} svc_fh;

typedef enum { inactive, active } mutex;

/*
 * Paths constructed in this system always consist of real directories
 * (excepting the last element) i.e. they do not contain symbolic links.
 * This is guaranteed by the way PDFS constructs the paths.
 * As a consequence we may assume that
 *	/x/y/z/.. == /x/y
 * and	/x/y/z/. == /x/y/z
 * provided that z != . && z != ..
 * These relations are exploited in fh_compose.
 *
 * Further assumptions:
 *	All cached pathnames consist of a leading /
 *	followed by zero or more / separated names
 *	s.t.
 *		name != .
 *		name != ..
 *		index(name, '/') == 0
 */
typedef struct fhcache {
	struct fhcache *	next;
	struct fhcache *	prev;
	struct fhcache *	hash_next;
	struct fhcache *	fd_next;
	struct fhcache *	fd_prev;
	svc_fh			h;
	int			fd;
	int			omode;
	char *			path;
	time_t			last_used;
	// pdfs_client *	last_clnt;
	// pdfs_mount *		last_mount;
	uid_t			last_uid;
	int			flags;
} fhcache;

/* Global FH variables. */
extern int			fh_initialized;

/* function prototypes. */

pdfsstat pdfs_errno(void);
psi_t	pseudo_inode(ino_t inode, dev_t dev);
void	fh_init(void);
char	*fh_pr(pdfs_fh *fh);
int	fh_create(pdfs_fh *fh, char *path);
fhcache	*fh_find(svc_fh *h, int create);
char	*fh_path(pdfs_fh *fh, pdfsstat *status);
int	path_open(char *path, int omode, int perm);
int	fh_fd(fhcache *fhc, pdfsstat *status, int omode);
void	fd_inactive(int fd);
pdfsstat fh_compose(struct diropargs *dopa, pdfs_fh *new_fh, 
	struct stat **sbpp, int fd, int omode, int public);
psi_t	fh_psi(pdfs_fh *fh);
void	fh_remove(char *path);
pdfs_fh	*fh_handle(fhcache *fhc);
void	fh_flush(int force);
void flush_cache(int sig);
int	pdfsmounted(const char *path, struct stat *sbp);

extern void log_write(char *, ...);
#endif

/* End of fh_cache.h. */

