/*
 *  linux/fs/nfs/proc.c
 *
 *  Copyright (C) 1992, 1993, 1994  Rick Sladkey
 *
 *  OS-independent nfs remote procedure call functions
 *
 *  Tuned by Alan Cox <A.Cox@swansea.ac.uk> for >3K buffers
 *  so at last we can have decent(ish) throughput off a 
 *  Sun server.
 *
 *  Coding optimized and cleaned up by Florian La Roche.
 *  Note: Error returns are optimized for NFS_OK, which isn't translated via
 *  nfs_stat_to_errno(), but happens to be already the right return code.
 *
 *  FixMe: We ought to define a sensible small max size for
 *  things like getattr that are tiny packets and use the
 *  old get_free_page stuff with it.
 *
 *  Also, the code currently doesn't check the size of the packet, when
 *  it decodes the packet.
 *
 *  Feel free to fix it and mail me the diffs if it worries you.
 *
 *  Completely rewritten to support the new RPC call interface;
 *  rewrote and moved the entire XDR stuff to xdr.c
 *  --Olaf Kirch June 1996
 */

#define NFS_NEED_XDR_TYPES

#include <linux/param.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/utsname.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/in.h>
#include <linux/pagemap.h>
#include <linux/sunrpc/clnt.h>
#include <linux/nfs3_fs.h>

#include <asm/segment.h>

#ifdef NFS_DEBUG
# define NFSDBG_FACILITY	NFSDBG_PROC
#endif

extern int
nfs_xdr_fhandle(struct rpc_rqst *req, u32 *p, struct nfs_fh3 *fh);  

extern u32 *
xdr_decode_fattr(u32 *p, struct fattr3 *fattr);   

/*
 * One function for each procedure in the NFS protocol.
 */
int
nfs_proc_getattr(struct nfs_server *server, struct nfs_fh3 *fhandle,
			struct fattr3 *fattr)
{
	int	status;

	// printk("proc: NFS V3 call  getattr\n");
	status = rpc_call(server->client, NFSPROC_GETATTR, fhandle, fattr, 0); 
	
	// printk("proc: NFS V3 reply getattr status is %d\n", status);
	return status;
}

int
nfs_proc_setattr(struct nfs_server *server, struct nfs_fh3 *fhandle,
struct setattr3 *sattr, struct fattr3 *fattr, struct nfs3_wcc_attr *wccattr)
{
	struct nfs3_sattrargs	arg = { fhandle, sattr };
	struct nfs3_wcc_data	res = { wccattr, fattr }; 
	int	status;
	
	// printk("NFS call  setattr\n");
	status = rpc_call(server->client, NFSPROC_SETATTR, &arg, &res, 0);
	// printk("NFS reply setattr\n");
	return status;
}

int
nfs_proc_lookup(struct nfs_server *server, struct nfs_fh3 *dir, 
		const char *name,
		    struct nfs_fh3 *fhandle, struct fattr3 *fattr)
{
	struct nfs3_diropargs	arg = { dir, name };
	struct nfs3_diropok	res = { fhandle, fattr };
	int			status;

	// printk("NFS call  lookup %s\n", name);
	status = rpc_call(server->client, NFSPROC_LOOKUP, &arg, &res, 0);
	// printk("NFS reply lookup: %d\n", status);
	return status;
}

int
nfs_proc_readlink(struct nfs_server *server, struct nfs_fh3 *fhandle,
			void **p0, char **string, unsigned int *len,
			unsigned int maxlen)
{
	/* struct nfs_readlinkres	res = { string, len, maxlen, NULL }; */
	int			status;

	/* dprintk("NFS call  readlink\n");
	status = rpc_call(server->client, NFSPROC_READLINK, fhandle, &res, 0);
	dprintk("NFS reply readlink: %d\n", status);
	if (!status)
		*p0 = res.buffer;
	else if (res.buffer)
		kfree(res.buffer); */
	return status;
} 

int
nfs_proc_read(struct nfs_server *server, struct nfs_fh3 *fhandle, int swap,
			  unsigned long offset, unsigned int count,
			  void *buffer, struct fattr3 *fattr)
{
	struct nfs3_readargs	arg = { fhandle, offset, count, buffer};
	struct nfs3_readres	res = { fattr, count }; 
	int			status;

	// printk("NFS call  read %d @ %ld\n", count, offset);
	status = rpc_call(server->client, NFSPROC_READ, &arg, &res,
			swap? NFS_RPC_SWAPFLAGS : 0); 
	// printk("NFS reply read: %d\n", status);  
	
	return status;
}



 
int
nfs_proc_write(struct nfs_server *server, struct nfs_fh3 *fhandle, int swap,
			unsigned long offset, unsigned int count,
			const void *buffer, struct fattr3 *fattr)
{
	struct nfs3_writeargs	arg = { fhandle, offset, count, buffer };
	int			status;

	// printk("NFS call  write %d @ %ld\n", count, offset);
	status = rpc_call(server->client, NFSPROC_WRITE, &arg, fattr,
			swap? (RPC_TASK_SWAPPER|RPC_TASK_ROOTCREDS) : 0);
	// printk("NFS reply read: %d\n", status);
	return status < 0? status : count; 

}



int
nfs_proc_create(struct nfs_server *server, struct nfs_fh3 *dir,
			const char *name, struct setattr3 *sattr,
			struct nfs_fh3 *fhandle, struct fattr3 *fattr)
			
{
	struct nfs3_createargs	arg = { dir, name, sattr };
	struct nfs3_createres	res = { fhandle, fattr };
	int			status;

	// printk("NFS call  create %s\n", name);
	status = rpc_call(server->client, NFSPROC_CREATE, &arg, &res, 0);
	// printk("NFS reply create: %d\n", status);
	return status;
}

int
nfs_proc_remove(struct nfs_server *server, struct nfs_fh3 *dir, const char *name, nfs3_wcc_attr *wccattr, fattr3 *fattr)
{
	struct nfs3_diropargs	arg = { dir, name };
	struct nfs3_wcc_data	res = { wccattr, fattr };
	int			status;

	// printk("NFS call  remove %s\n", name);
	status = rpc_call(server->client, NFSPROC_REMOVE, &arg, &res, 0);
	// printk("NFS reply remove: %d\n", status);
	return status;
}

int
nfs_proc_rename(struct nfs_server *server,
		struct nfs_fh3 *old_dir, const char *old_name,
		struct nfs_fh3 *new_dir, const char *new_name,
		struct nfs3_wcc_data *fromattr, struct nfs3_wcc_data *toattr)
{
	struct nfs3_renameargs	arg = { old_dir, old_name, new_dir, new_name };
	struct nfs3_renameres	res = { fromattr, toattr };
	int			status;

	// printk("NFS call  rename %s -> %s\n", old_name, new_name);
	status = rpc_call(server->client, NFSPROC_RENAME, &arg, &res, 0);
	// printk("NFS reply rename: %d\n", status);
	return status;
}

int
nfs_proc_link(struct nfs_server *server, struct nfs_fh3 *fhandle,
		struct nfs_fh3 *dir, const char *name,
		fattr3 *fattr, nfs3_wcc_attr *dirwccattr, fattr3 *dirattr)
{
	struct nfs3_linkargs	arg = { fhandle, dir, name };
	struct nfs3_linkres	res = {	fattr, dirwccattr, dirattr };
	int			status;

	// printk("NFS call  link %s\n", name);
	status = rpc_call(server->client, NFSPROC_LINK, &arg, &res, 0);
	// printk("NFS reply link: %d\n", status);
	return status;
}


int
nfs_proc_symlink(struct nfs_server *server, struct nfs_fh3 *dir,
			const char *name, const char *path,
			struct setattr3 *sattr, nfs_fh3 *fhandle,
			fattr3	*fattr)
{
	struct nfs3_symlinkargs	arg = { dir, name, path, sattr }; 
	struct nfs3_createres	res = { fhandle, fattr };
	int			status;

	// printk("NFS call  symlink %s -> %s\n", name, path);
	status = rpc_call(server->client, NFSPROC_SYMLINK, &arg, &res, 0);
	// printk("NFS reply symlink: %d\n", status); 
	return status;
}



int
nfs_proc_mkdir(struct nfs_server *server, struct nfs_fh3 *dir,
			const char *name, struct setattr3 *sattr,
			struct nfs_fh3 *fhandle, struct fattr3 *fattr)
{
	struct nfs3_createargs	arg = { dir, name, sattr };
	struct nfs3_diropok	res = { fhandle, fattr };
	int			status;

	// printk("NFS call  mkdir %s\n", name);
	status = rpc_call(server->client, NFSPROC_MKDIR, &arg, &res, 0);
	// printk("NFS reply mkdir: %d\n", status);
	return status;
}

int
nfs_proc_rmdir(struct nfs_server *server, struct nfs_fh3 *dir, const char *name,nfs3_wcc_attr *wcc_attr, fattr3 *fattr)
{
	struct nfs3_diropargs	arg = { dir, name };
	struct nfs3_wcc_data	res = { wcc_attr, fattr };
	int			status;

	// printk("NFS call  rmdir %s\n", name);
	status = rpc_call(server->client, NFSPROC_RMDIR, &arg, &res, 0);
	// printk("NFS reply rmdir: %d\n", status);
	return status;
}

/*
 * The READDIR implementation is somewhat hackish - we pass a temporary
 * buffer to the encode function, which installs it in the receive
 * iovec. The dirent buffer itself is passed in the result struct.
 */
int
nfs_proc_readdir(struct nfs_server *server, struct nfs_fh3 *fhandle,
	u32 cookie, unsigned int size, __u32 *entry, struct fattr3 *dirfattr) 

{
	struct nfs3_readdirargs	arg;
	struct nfs3_readdirres	res;
	void *			buffer;
	unsigned int		buf_size = PAGE_SIZE;
	int			status;

	/* First get a temp buffer for the readdir reply */
	/* N.B. does this really need to be cleared? */
	status = -ENOMEM;
	buffer = (void *) get_free_page(GFP_KERNEL);
	if (!buffer)
		goto out;

	/*
	 * Calculate the effective size the buffer.  To make sure
	 * that the returned data will fit into the user's buffer,
	 * we decrease the buffer size as necessary.
	 *
	 * Note: NFS returns three __u32 values for each entry,
	 * and we assume that the data is packed into the user
	 * buffer with the same efficiency. 
	 */
	if (size < buf_size)
		buf_size = size;
	if (server->rsize < buf_size)
		buf_size = server->rsize;
#if 1
printk("nfs_proc_readdir: user size=%d, rsize=%d, buf_size=%d\n",
size, server->rsize, buf_size);
#endif

	arg.fh = fhandle;
	arg.cookie = cookie;
	// printk("arg.cookie is %ld\n", cookie);
	arg.buffer = buffer;
	arg.bufsiz = buf_size;
	res.fattr = dirfattr;
	res.buffer = entry;
	res.bufsiz = size;

	// printk("NFS call  readdir %d\n", cookie);
	status = rpc_call(server->client, NFSPROC_READDIR, &arg, &res, 0);
	// printk("NFS reply readdir: %d\n", status);
	free_page((unsigned long) buffer);
out:
	return status;
}

int
nfs_proc_fsinfo(struct nfs_server *server, struct nfs_fh3 *fhandle,
			struct nfs3_fsinfo *info)
{
	int	status;

	// printk("NFS call fsinfo\n");
	status = rpc_call(server->client, NFSPROC_FSINFO, fhandle, info, 0);
	// printk("NFS reply fsinfo: %d\n", status);
	return status;
}


int 
nfs_proc_fsstat (struct nfs_server *server, struct nfs_fh3 *fhandle,
					struct nfs3_fsstat *fsstat)
{

	int 	status;

	// printk("NFS call fsstat\n");
	status = rpc_call(server->client, NFSPROC_FSSTAT, fhandle, fsstat, 0);
	// printk("NFS reply fsstat: %d\n", status);

	return status;	
}

int 
nfs_proc_mknod(struct nfs_server *server, struct nfs_fh3 *dir,
			const char *name, struct setattr3 *sattr,
			struct nfs_fh3 *fhandle, struct fattr3 *fattr)
{
    int	status;
    struct nfs3_createargs	arg = { dir, name, sattr };
    struct nfs3_createres	res = { fhandle, fattr };
	
    // printk("NFS call mknod\n");
    status = rpc_call(server->client, NFSPROC_MKNOD, &arg, &res, 0);
    // printk("NFS reply mknod: %d\n", status);
   
    return status;
}                                                                                      
int 
nfs_proc_readdirplus (struct nfs_server *server, struct nfs3_rdplusargs 
*rdplusargs, struct nfs3_rdplusres *rpsres)
{
        int status;

    	// printk("NFS call rdplus\n");
        status = rpc_call(server->client, NFSPROC_READDIRPLUS, rdplusargs, rpsres, 0);
        // printk("NFS reply readdirplus: %d\n", status);
	return status;
}

int 
nfs_proc_commit (struct nfs_server *server, struct nfs3_commitargs *commitargs, 
struct nfs3_commitres *ctres)
{
	int	status;
	
        // printk("NFS call commit\n");
        status = rpc_call(server->client, NFSPROC_COMMIT, commitargs, ctres, 0);
        // printk("NFS reply commit: %d\n", status);
		
	return 	status;
}

int 
nfs_proc_pathconf (struct nfs_server *server, struct nfs_fh3 *fhandle, 
struct nfs3_pathconfres *pfres)
{
	int 	status;
	
        // printk("NFS call pathconf\n");
        status = rpc_call(server->client, NFSPROC_PATHCONF, fhandle, pfres, 0);
        // printk("NFS reply pathconf: %d\n", status);
	return 	status;
}      
