/**
*    Name:    Menu.java
*    Author:  Lei Shen
*    Date:    Feb. 16, 1999
*
*    Starting Environment for using SDatr tools.
*/
import java.io.*;
import java.util.*;

class Menu 
{
   /*
   *  A hashtable for quickly looking up command available
   */
   private Hashtable   hashCommand;

   /*
   *  A parser for parsing datr source file
   */
   private SDatrParser parser;

   /*
   *  A place for storing internal information about the parsed datr file
   */
   private Theory      theory;

   /*
   *  Stores the name of previously compiled DATR file
   */
   private String  prevFileName = null;


   /**
   *  In order to add a command to the system
   *  1) add an entry here into the hashCommand Table
   *  2) add a case statement in function start() 
   *     to tell which function to call for this command
   *  3) implement this function call
   */
   public Menu()
   {
      hashCommand = new Hashtable();
      hashCommand.put( "help",    new Integer(1) );
      hashCommand.put( "exit",    new Integer(2) );
      hashCommand.put( "quit",    new Integer(2) );
      hashCommand.put( "compile", new Integer(3) );
      hashCommand.put( "cp",      new Integer(4) );
      hashCommand.put( "augment", new Integer(5) );
      hashCommand.put( "query",   new Integer(6) );
      hashCommand.put( "debug",   new Integer(7) );
      hashCommand.put( "theory",  new Integer(8) );
      hashCommand.put( "leaves",  new Integer(9) );
      hashCommand.put( "augleaves",new Integer(10) );
      hashCommand.put( "analyze", new Integer(11) );
      hashCommand.put( "list",    new Integer(12) );
      hashCommand.put( "warndup", new Integer(13) );
      parser = new SDatrParser(System.in);
      theory = new Theory();
   }

/*****************************************************************************
                    Those command implementations.
*****************************************************************************/
   /**
   *  Print out greet information
   */
   private void funcGreeting()
   {
      System.err.println( "Welcome To KATR 1.0" );
      System.err.println( "Author:    Lei  Shen" );
      System.err.println( "Date:      Dec. 1998" );
   }

   /**
   *  Print out information when an unexpected bug appears
   */
   static private void funcBugs()
   {
      System.out.println( "Please report this bug to:" );
      System.out.println( "Lei Shen, lshen0@engr.uky.edu" );
   }

   /**
   *  Print out help information 
   */
   private void funcHelp( StringTokenizer tokensMother )
   {
      if( tokensMother != null && tokensMother.hasMoreTokens() ) 
      {
         System.out.println( "Usage: help/0" );
      }
      else {
         System.err.println( "SDatr Help" );
         System.err.println( "help              -- this screen" );
         System.err.println( "compile filename  -- compile the datr file" );
         System.err.println( "cp                -- recompile the current datr file" );
         System.err.println( "augment filename  -- augment datr code" );
         System.err.println( "query   filename  -- query the query file" );
         System.err.println( "debug   level     -- set debug level" );
         System.err.println( "        level 0   -- no debug"  );
         System.err.println( "        level 1   -- debug while parsing"  );
         System.err.println( "        level 2   -- debug while querying"  );
         System.err.println( "theory            -- perform all the theory" );
         System.err.println( "leaves            -- perform theory on leaves" );
         System.err.println( "augleaves         -- perform theory on augmented leaves" );
         System.err.println( "analyze           -- list certain warnings" );
         System.err.println( "list              -- dump theory" );
         System.err.println( "warndup true/false     -- set debug level" );
         System.err.println( "exit -- exit KATR " );
      }
   }

   /**
   *   exit the system
   */
   private void funcExit( StringTokenizer tokensMother )
   {
      if( tokensMother.hasMoreTokens() ) 
         System.out.println( "Usage: exit/0" );
      else 
         System.exit( 0 );
   }

   /**
   *   compile datrfile
   */
   private void funcCompile( StringTokenizer tokensMother) 
   {
      if( tokensMother.countTokens() != 1 )
          System.out.println( "Usage: compile/1 filename" );
      else
      {
          String  strFileName = tokensMother.nextToken();
          prevFileName = strFileName;
          try
          {
             theory.clearTheory();
             parser.ReInit(new java.io.FileInputStream(strFileName) );
             try
             {
                parser.ParseTheory( theory );
             }
             catch( ParseException e )
             {
                System.out.println( e );
             }
          } 
          catch (java.io.FileNotFoundException e) 
          {
              System.out.println( "File " + strFileName + " not found.");
          }
      }
   }

   /**
   *   recompile datrfile
   */
   private void funcRecompile( StringTokenizer tokensMother) 
   {
      if( tokensMother.countTokens() != 0 )
          System.out.println( "Usage: cp/0" );
      else if( prevFileName == null)
          System.out.println(" Run compile command first\n");
      else
      {
          System.out.println ("compile "+prevFileName);
          String  strFileName = prevFileName;
          try
          {
             theory.clearTheory();
             parser.ReInit(new java.io.FileInputStream(strFileName) );
             try
             {
                parser.ParseTheory( theory );
             }
             catch( ParseException e )
             {
                System.out.println( e );
             }
          } 
          catch (java.io.FileNotFoundException e) 
          {
              System.out.println( "File " + strFileName + " not found.");
          }
      }
   }

   /**
   *   augment compiled datr code dynamically
   */
   private void funcAugment( StringTokenizer tokensMother) 
   {
      if( tokensMother.countTokens() != 1 )
          System.out.println( "Usage: augment/1 filename" );
      else
      {
          String  strFileName = tokensMother.nextToken();
          prevFileName = strFileName;
          try
          {
             if(strFileName.equals("stdin"))
                parser.ReInit(System.in);
             else
                parser.ReInit(new java.io.FileInputStream(strFileName) );
             try
             {
                parser.ParseAugment( theory );
             }
             catch( ParseException e )
             {
                System.out.println( e );
             }
          } 
          catch (java.io.FileNotFoundException e) 
          {
              System.out.println( "File " + strFileName + " not found.");
          }
      }
   }

   /**
   *   query datrqueryfile
   */
   private void funcQuery( StringTokenizer tokensMother ) 
           throws ParseException
   {
      if( tokensMother.countTokens() != 1 )
          System.out.println( "Usage: query/1 filename" );
      else
      {
          String  strFileName = tokensMother.nextToken();
          try
          {
             if(strFileName.equals("stdin"))
                parser.ReInit(System.in);
             else
                parser.ReInit(new java.io.FileInputStream(strFileName) );
             try
             {
                parser.ParseQuery( theory );
             }
             catch( ParseException e )
             {
                System.out.println( e );
             }
          } 
          catch (java.io.FileNotFoundException e) 
          {
              System.out.println( "File " + strFileName + " not found.");
          }
      }
   }

   /**
   *   set debug level
   */
   private void funcDebug( StringTokenizer tokensMother )
   {
     if( tokensMother.countTokens() != 1 )
         System.out.println( "Usage: debug/1 level" );
     else
     {
          String  strLevel = tokensMother.nextToken();
          
          try
          {
             Long lLevel = new Long( strLevel );
             theory.setDebugLevel( lLevel.longValue() );
          }
          catch( NumberFormatException e )
          {
             System.out.println( e );
          }
     }
   }

   /**
   *   dump out all the theory
   */
   private void funcTheory( StringTokenizer tokensMother )
   {
     if( tokensMother.countTokens() != 0 )
         System.out.println( "Usage: theory/0" );
     else
     {
         try
         {
            theory.showTheory();
         }
         catch(Exception e )
         {
            System.out.println( e );
         }
     }
   }

   /**
   *   show analyze
   */
   private void funcAnalyze( StringTokenizer tokensMother )
   {
     if( tokensMother.countTokens() != 0 )
         System.out.println( "Usage: analyze/0" );
     else 
     {
        try
        {
            theory.analyze();
        }
        catch (Exception e)
        {
            System.out.println( e );
        }
     }
   }

   /**
   *   show leaves
   */
   private void funcLeaves( StringTokenizer tokensMother )
   {
     if( tokensMother.countTokens() != 0 )
         System.out.println( "Usage: leaves/0" );
     else
     {
	 try
	 {
	     theory.showLeaves();
	 }
	 catch(Exception e )
	 {
	    System.out.println( e );
	 }
     }
   }

   /**
   *   show theory on newly agumented leaves ie., leaves augmented using the
   *   augment command.
   */
   private void funcAugleaves( StringTokenizer tokensMother )
   {
     if( tokensMother.countTokens() != 0 )
         System.out.println( "Usage: augleaves/0" );
     else
     {
	 try
	 {
	     theory.showAugleaves();
	 }
	 catch(Exception e )
	 {
	    System.out.println( e );
	 }
     }
   }


   /**
   *   list the datr file represented internally
   */
   private void funcList( StringTokenizer tokensMother )
   {
     if( tokensMother.countTokens() != 0 )
         System.out.println( "Usage: list/0" );
     else {
          try {
             System.out.println(theory.toString());
          } catch(NumberFormatException e){
             System.out.println( e );
          }
     }
   }

   private void funcWarnDup( StringTokenizer tokensMother )
   {
     if( tokensMother.countTokens() != 1 )
         System.out.println( "Usage: warndup true/false" );
     else {
          String  strWarn = tokensMother.nextToken();
          if("false".equals(strWarn))
             theory.setWarnDuplicate(false);
          else if("true".equals(strWarn))
             theory.setWarnDuplicate(true);
     }
   }


/***************************************************************************
               Command interpretor
***************************************************************************/
   private String prompt = "> ";

   public  void start() throws Exception
   {
      funcGreeting();
      funcHelp( null );

      String          strCommandLine = null;
      String          token = null;
      StringTokenizer tokensMother = null;
      Integer         intCommand = null;
      BufferedReader  input = new BufferedReader( 
                                   new InputStreamReader(System.in) );
      while( true )
      {
         System.out.print( prompt );
         strCommandLine   = input.readLine();
         tokensMother = new StringTokenizer( strCommandLine );
         while( tokensMother.hasMoreTokens() )
         {
            token = tokensMother.nextToken();
            intCommand = (Integer) hashCommand.get( token );
            if( intCommand != null ) 
            {
               switch( intCommand.intValue() )
               {
                  case 1:  funcHelp( tokensMother );
                           break;
                  case 2:  funcExit( tokensMother );
                           break;
                  case 3:  funcCompile( tokensMother );
                           break;
                  case 4:  funcRecompile( tokensMother );
                           break;
                  case 5:  funcAugment( tokensMother );
                           break;
                  case 6:  funcQuery( tokensMother );
                           break;
                  case 7:  funcDebug( tokensMother );
                           break;
                  case 8:  funcTheory( tokensMother );
                           break;
                  case 9:  funcLeaves( tokensMother );
                           break;
                  case 10:  funcAugleaves( tokensMother );
                           break;
                  case 11:  funcAnalyze( tokensMother );
                           break;
                  case 12:  funcList( tokensMother );
                           break;
                  case 13:  funcWarnDup( tokensMother );
                           break;
                  default: funcBugs();
                           break;
               }
            }
            else 
            {
               System.out.println( "Unrecognized Command" );               
               System.out.println( "Use help for a command list" );
            }
            break;
         }
      }
   }

   /**
   *  Entry point for the whole program
   */
   public static void main( String args[] )
   {
      try
      {
         Menu  menu = new Menu();
         menu.start();
      }
      catch( Exception e )
      {
         System.out.println( "In Main:" + e );
         e.printStackTrace();
         funcBugs();
      }
   }
}
