/* JOBPOOL.H  */

/* Declaration of classes */
/* Member functions are defined in JOBPOOL.C  */

#ifndef JOBPOOL_H
#define JOBPOOL_H

#define MAXAGENTNUMBER 512
#define MAXMACHINENUMBER 128
#define MAXJOBNUMBER  32768
#define LENGTHOFHOSTNAME 64
#define LENGTHOFJOBNAME 128
#define MAXDATASECTIONNUMBER 64
#define MAXSTRLENGTH 256

#define MAINPROGRAMADDRS -9999
#define REQUEST 0
#define RESULT 1
#define CONFIRMATION 2

typedef int Info;
typedef int Tid;
typedef int JobID;
typedef int JobNumber;
typedef int ResultID;
typedef int AgentID;
typedef int MachineID;
typedef int Msgtag;

/* PackageType represents the type of package: request package or
 result package.  */
typedef int PackageType;
 
/* CheckTime controls the time interval in seconds between which the living
 status is checked for all of the machines in a virtual machine.  */
typedef int CheckTime;

/* StorageType decides whether Buffer is created in place or somewhere else. */
enum    StorageType { OUTOF_PLACE, IN_PLACE };

/* JobStatus represents the status of a job in the jobpool. */
enum    JobStatus   { INITIAL, SENT, FINISHED, SENTTOSELF };

/* RunType decides how many JobAgents are spawned in a single machine. */ 
enum    RunType     { NICE, NORMAL, FAST };

/* MachineStatus denotes the status of a machine in the 'hostname' file. */
enum    MachineStatus { ALIVE, SPAWNED, DEAD };

/* Reliability decides the tolerance for crashing of machines or networks. */ 
enum    Reliability { LOW, HIGH };

/* RunMode controls the mode of starting a virtual machine. In MANUAL mode,
 the virtual machine has to be started before an application runs. */
enum    RunMode { MANUAL, AUTOSTART };

/* AddMode controls the mode of add a host into a group. In NEW mode, only
 a new host is added into group. In ANY mode, any host can be added into
 group. (AddMode is only used in Class GroupManager) */
enum    AddMode { NEW, ANY };
 
/* Buffer is used to temporarily store data to be delivered. */
typedef struct {
      enum StorageType BufferType;
      int *intptr[MAXDATASECTIONNUMBER];
      int intnum[MAXDATASECTIONNUMBER];
      int intcnt;
      int intrev;
      float *floatptr[MAXDATASECTIONNUMBER];
      int floatnum[MAXDATASECTIONNUMBER];
      int floatcnt; 
      int floatrev;
      char *strptr[MAXDATASECTIONNUMBER];
      int strnum[MAXDATASECTIONNUMBER];
      int strcnt; 
      int strrev; 
} Buffer;

/* Functions about Buffer */
      void InitBuffer( Buffer *, enum StorageType );
      void PackInt(Buffer *, int *, int);
      void PackFloat(Buffer *, float *, int);
      void PackString(Buffer *, char *);
      void UnpackInt(Buffer *, int *, int);
      void UnpackFloat(Buffer *, float *, int);
      void UnpackString(Buffer *, char *);


/* Machine is for the infomation about host machines.  */
typedef struct {
     char *name;
     MachineID  machineID;
     int  jobID[MAXJOBNUMBER];
     enum MachineStatus status;
     int  jobCount;
} Machine;

/* Functions about Machines */
     void InitMachine(Machine *);
     void InsertJobID(Machine *, JobID);
     int  RemoveJobID(Machine *, JobID);

/* Agent records the information about a remote JobAgent. */ 
typedef struct{
      char *hostName;
      char *hostArch;
      int  hostSpeed;
      Tid  myTid;
      Tid  parentId;
      JobID  jobID;
} AgentInfo;

/* Job encapsulates all data and other information about a task to be 
 parallelized. */
typedef struct {
      PackageType type;
      Tid         source;
      Tid         destination;

      Buffer *broadcastReceiveBuffer;
      Buffer *privateReceiveBuffer;
      Buffer *privateSendBuffer;

      enum JobStatus jobStatus;
      AgentID    agentID;
      MachineID machineID;
      JobID jobID;
      JobID localJobID;
      char *jobName;
} Job;

/* Functions about Job */
      void InitJob( Job *, enum StorageType );
      void SetJobName(Job *, char *);
      void SetLocalJobID(Job *, JobID );
      JobID  GetJobID(Job *);
      JobID  GetLocalJobID(Job *);
      char* GetJobName(Job *);
      void SetJobID( Job *, JobID );
      
/* Result contains the results from a requested job.  */
typedef struct {
      PackageType type;
      Tid         source;
      Tid         destination;

      Buffer *broadcastReceiveBuffer;
      Buffer *privateReceiveBuffer;
      Buffer *privateSendBuffer;

      ResultID resultID;
      ResultID localResultID;
      char *resultName;

} Result;

/* Functions about Result */
      void InitResult( Result *, enum StorageType ); 
      void SetResultName(Result *, char *);
      void SetResultID(Result *, JobID);
      void SetLocalResultID(Result *, JobID);
      ResultID  GetResultID(Result *);
      ResultID  GetLocalResultID(Result *);
      char *GetResultName(Result *);
      
/* JobPool is a collection of jobs to be parallelized.  */
typedef struct{
      JobID     nextJobID;
      JobNumber jobNumber;
      JobNumber totalJobNum;
      JobNumber userJobNum;
      Job *job[MAXJOBNUMBER];
} JobPool;

/* Functions of JobPool */
      void InitJobPool( JobPool *, Job *, int, enum StorageType );
      Info AddJob( JobPool *, Job *, enum StorageType );
      void RemoveJob( JobPool *, JobID );
      JobNumber GetJobNumber(JobPool *);
      JobNumber GetTotalJobNumber(JobPool *);
      JobNumber GetUserJobNumber(JobPool *);
      Job *GetJob(JobPool *, JobNumber);
      void ReduceJobNumberByOne(JobPool *);
      void ReduceUserJobNumberByOne(JobPool *);

/* JobManager administrates the whole parallelizing computation process. */ 
typedef struct {
      JobPool     *jobPool;
      enum RunType     runType;
      enum Reliability reliability;
      AgentID     nextAgentID;
      MachineID   nextMachineID;
      Buffer      *broadcastBuffer;
      AgentInfo   *agent[MAXAGENTNUMBER];
      char        *hostName[MAXMACHINENUMBER];
      Result      *result;
      JobID       resendJobID;
      Machine     *machines;
      enum RunMode  runMode;
      CheckTime   checktime;
} JobManager;

/* Functions of JobManager */
      void InitJobManager(JobManager *, Job *, JobNumber, Buffer *, 
			  enum RunType, enum Reliability, enum RunMode,
			  CheckTime);
      void    EndJobManager(JobManager *);
      Result* RunJobManager(JobManager *);
      int     SendToNewHost(JobManager *, MachineID, int, enum JobStatus);
      int     Resend(JobManager *, MachineID, AgentID, JobID, enum JobStatus);

/* JobAgent is a remote representative sitting in a worker machine and
 delivers data between JobManager and its Worker.  */
typedef struct {
      Tid myTid;
      Tid parentTid;
      PackageType currentType;
      Tid currentSource;
      Tid currentDestination;
      JobID currentJobID;
      JobID localJobID;
} JobAgent;

/* Functions of JobAgent */
      void InitJobAgent(JobAgent *);
      void RunJobAgent( JobAgent *, Result* (*Worker)(Job *) );
      Job* Receive(JobAgent *);
      void Send(JobAgent *, Result *);

/* GroupManager administrates the parallelizing computation process of
 one group. */ 
typedef struct {
      JobPool     *jobPool;
      enum RunType     runType;
      AgentID     nextAgentID;
      MachineID   nextMachineID;
      Buffer      *broadcastBuffer;
      AgentInfo   *agent[MAXAGENTNUMBER];
      char        *hostName[MAXMACHINENUMBER];
      Result      *result;
      Machine     *machines;
      enum AddMode     addMode;
} GroupManager;

/* Functions of GroupManager  */
      void InitGroupManager(GroupManager *, Job *, JobNumber, Buffer *,
			    enum RunType, enum AddMode);
      Result* RunGroupManager(GroupManager *);

typedef struct {
      JobPool     *jobPool;
      Buffer      *broadcastBuffer;
      Result      *result;
} Distributor;

/* Functions of Distributor */
      void InitDistributor(Distributor *, Job *, JobNumber, Buffer *);
      Result* RunDistributor(Distributor *);

// Utility function for packing Buffer into PVM buffer.
void PackBuffer(Buffer *);

// Utility function for unpacking Buffer from PVM buffer.
void UnpackBuffer(Buffer *);

#endif 
