/* 
 * grid.c
 * copyright (c) 2004 Wei-Keat Kong.
 *
 * Functions for loading a crossword grid from file.
 *
 * CellInit: Initializes grid cell
 * load_gridCMD: Tk interface to load the grid
 * setup_gridCMD: Tk interface to set up the grid
*/

#include "grid.h"
#include <stdio.h>

/*
 * Function: CellInit
 * Purpose: analyze each cell, determine if it has a constraint, and
 *          if it is the beginning of a word.
 * I/O: In  - x,y of the grid
 *      Out - index of the slotlist
 * Returns: none.
 * Functions called: none.
 * Algorithm: read all down constraints first, then read across. 
*/
void CellInit(u_short x, u_short y, u_short *slotlist_idx)
{
   u_short across=0; // across clue?
   u_short down=0;   // down clue?
   u_short length=0;
   u_short offset_idx=0;
   u_short arrayi;

   // Validate input
   if((x>=GRIDSIZE)||(y>=GRIDSIZE))
   {
      fprintf(stderr, "CellInit: Invalid x,y. %d,%d\n", x,y);
      exit(RC_FAILED);
   }

   if(!grid[x][y].isblack)
   {
      // Checks if cell below is non-black. If so, then it is a down clue.
      if(x==0)
      {
         if(!grid[x+1][y].isblack) down=1;
      }
      else
      {
         if(x==GRIDSIZE-1) down=0;
         else
         {
            if((grid[x-1][y].isblack)&&
               (!grid[x+1][y].isblack)) down=1;
         }
      }

      if(down==1)
      {
         // this cell is a down clue
         length=0;
         for(offset_idx=x;offset_idx<GRIDSIZE;offset_idx++)
         {
            if(!grid[offset_idx][y].isblack)
            {
               if(y==0)
               {
                  if(!grid[offset_idx][y+1].isblack)
                  {
                     grid[offset_idx][y].constrained = 1;
                  }

               }
               else
               {
                  if(y==GRIDSIZE-1)
                  {
                     if(!grid[offset_idx][y-1].isblack)
                     {
                       grid[offset_idx][y].constrained = 1;
                     }
                  }
                  else
                  {
                     if((!grid[offset_idx][y+1].isblack)||
                       (!grid[offset_idx][y-1].isblack))
                     {
                         grid[offset_idx][y].constrained = 1;
                     }
                  }
               }
               length++;
            }
            else break;
        }

         slotlist[*slotlist_idx].h_idx = x;
         slotlist[*slotlist_idx].w_idx = y;
         slotlist[*slotlist_idx].length = length;
         slotlist[*slotlist_idx].ucslength = 0;
         slotlist[*slotlist_idx].direction = DOWN;
         slotlist[*slotlist_idx].choices = 0;
         slotlist[*slotlist_idx].wordindex = -1;
         slotlist[*slotlist_idx].numconstraint = 0;
         for(arrayi=0;arrayi<GRIDSIZE;arrayi++)
         {
            slotlist[*slotlist_idx].constraint[arrayi].idx1 = 0;
            slotlist[*slotlist_idx].constraint[arrayi].pos1 = 0;
            slotlist[*slotlist_idx].constraint[arrayi].idx2 = 0;
            slotlist[*slotlist_idx].constraint[arrayi].pos2 = 0;
         }
         lengths[length-1]++;
         (*slotlist_idx)++;
      } // cell is a down clue

      // Checks if right cell is non-black. If so, then it is an across clue.
      if(y==0)
      {
         if(!grid[x][y+1].isblack) across=1; 
      }
      else 
      {
         if(y==GRIDSIZE-1) across=0;
         else 
         {
            if((grid[x][y-1].isblack)&&
               (!grid[x][y+1].isblack)) across=1;
         }
      }

      if(across==1)
      {
         // this cell is a across clue
         length=0;
         for(offset_idx=y;offset_idx<GRIDSIZE;offset_idx++)
         {
            if(!grid[x][offset_idx].isblack)
            {
               if(x==0)
               {
                  if(!grid[x+1][offset_idx].isblack)
                  {
                     grid[x][offset_idx].constrained = 1;
                  }
               }
               else
               {
                  if(x==GRIDSIZE-1)
                  {
                     if(!grid[x-1][offset_idx].isblack)
                     {
                        grid[x][offset_idx].constrained = 1;
                     }
                  }
                  else
                  {
                     if((!grid[x+1][offset_idx].isblack)||
                        (!grid[x-1][offset_idx].isblack))
                     {
                        grid[x][offset_idx].constrained = 1;
                     }
                  }
               }
               length++;
            }
            else break;
         }

         slotlist[*slotlist_idx].h_idx = x;
         slotlist[*slotlist_idx].w_idx = y;
         slotlist[*slotlist_idx].length = length; 
         slotlist[*slotlist_idx].ucslength = 0;
         slotlist[*slotlist_idx].direction = ACROSS;
         slotlist[*slotlist_idx].choices = 0;
         slotlist[*slotlist_idx].wordindex = -1;
         slotlist[*slotlist_idx].numconstraint = 0;
         for(arrayi=0;arrayi<GRIDSIZE;arrayi++)
         {
            slotlist[*slotlist_idx].constraint[arrayi].idx1 = 0;
            slotlist[*slotlist_idx].constraint[arrayi].pos1 = 0;
            slotlist[*slotlist_idx].constraint[arrayi].idx2 = 0;
            slotlist[*slotlist_idx].constraint[arrayi].pos2 = 0;
         }
         lengths[length-1]++;
         (*slotlist_idx)++;
      } // cell is an across clue
   } // cell is black
} // CellInit

/*
 * Function: load_gridCMD
 * Purpose: open a file, read its contents and initialize the global 
 *          structure grid.
 * I/O: In  - Standard Tk arguments
 *      Out - Standard Tk return codes
 * Returns: TCL_OK/TCL_FAILED
 * Functions called: WriteLog().
 * Algorithm: Read the grid size into global variable GRIDSIZE, 
 *            then initialize the grid.
*/
int load_gridCMD(ClientData clientdata, Tcl_Interp *interp, int argc,
                 char *argv[])
{
   FILE *fp=NULL; // File pointer
   int size=0, h_idx=0, w_idx=0, ucscnt=0;

   // Open argument 1 as file

   // fprintf(stderr, "opening grid %s\n", argv[1]);
   fp = fopen(argv[1], "r");
   if(fp == NULL)
   {
      WriteLog("load_grid(): Invalid file name\n");
      return TCL_ERROR;
   }

   // Read size of grid from file

   char buf[BUFSIZ];
   char *dummyPtr;
   dummyPtr = fgets(buf, BUFSIZ, fp);
   int dummy;
   dummy = sscanf(buf, "%d", &size);
   if(size > MAXGRIDSIZE+1) 
   {
      WriteLog("load_grid(): Invalid grid size %d\n", size);
      fclose(fp);
      return TCL_ERROR;
   }

   // Initialize global variable

   GRIDSIZE = size;
  
   // Initialize grid

   for(h_idx=0;h_idx<GRIDSIZE;h_idx++)
   {
      for(w_idx=0;w_idx<GRIDSIZE;w_idx++)
      {
         for(ucscnt=0;ucscnt<UCSLETTER;ucscnt++)
         {
            grid[h_idx][w_idx].letter[ucscnt] = 0;
         }
         grid[h_idx][w_idx].isblack = 0;
         grid[h_idx][w_idx].constrained = 0;
         grid[h_idx][w_idx].exists = 0;
      }
   }
   
   // Read black squares and update grid.

   while(fgets(buf, BUFSIZ, fp))
   {
   	  char initChar[10];
      int count = sscanf(buf, "%d %d %10s", &h_idx, &w_idx, initChar);
	  if (count == 2) { // a black square
	    // fprintf(stderr, "location (%d, %d) is black\n", h_idx, w_idx);
		grid[h_idx][w_idx].isblack = 1;
	  } else { // an initialized square
	    fprintf(stderr, "location (%d, %d) is initialized to %s\n",
		    h_idx, w_idx, initChar);
	    utf8_wide(grid[h_idx][w_idx].letter, initChar, UCSLETTER);
	    grid[h_idx][w_idx].exists = 1;
	  }
   }

   fclose(fp);
   return TCL_OK;
} // load_gridCMD

/*
 * Function: setup_gridCMD
 * Purpose: initialize the rest of the cells in the grid, create slotlist.
 * I/O: In  - Standard Tk arguments
 *      Out - Standard Tk return codes
 * Returns: TCL_OK/TCL_FAILED
 * Functions called: CelInit()
 * Algorithm: Initialize each cell, creating a word list.
 *            Initialize global variable NUM_WORDS.
*/
int setup_gridCMD(ClientData clientdata, Tcl_Interp *interp, int argc,
                 char *argv[])
{
   u_short height_idx=0,width_idx=0; // height and width
   u_short slotlist_idx=0;       // current index in the slotlist

   // Initialize each cell in the grid

   for(height_idx=0;height_idx<GRIDSIZE;height_idx++)
   {
      for(width_idx=0;width_idx<GRIDSIZE;width_idx++)
      {
         CellInit(height_idx, width_idx, &slotlist_idx);
      }
   }
   
   // Initialize global variable

   NUM_WORDS = slotlist_idx;

   // Find constraints for slotlist

   FindConstraintForSlotList(slotlist);

   return TCL_OK;
} // setup_gridCMD
