 /*! \file log_lib.c
 *  \brief This is the implementation of the Logging Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup logging_lib
*/


 /** \defgroup logging_lib The Logging Library
 *   \ingroup helper_components
 *   \brief Library that provides a method logging messages to stdout, stderr,
 *          syslogd, or a file.
 * 
 * Over the years, I have come to realize that logging is an extremely important
 * part of an application.  Inevitably, a situation arises when not every piece
 * of the process is configured properly to work with its environment.  Log
 * files are invaluable in situations of this nature.  When a catastrophe
 * occurs, the log gives you the information needed to avoid future problems.
 * With major processes in Linux, it is helpful to have this information
 * presented chronologically in the same file.  In Linux, this functionality is
 * provided by the syslog daemon (syslogd).
 *
 * In the early stages of development it is also very common to log activities.
 * Most developers use the occasional print statement that allows them to view
 * the current progress or state of the application.  Capturing all of the data
 * that defines the state of the program and then printing it, can often be
 * tedious.
 *
 * There are also situations where a developer might want to print the log
 * message to the screen and write to a file.  To perform this task by hand
 * would take twice as many lines of code.
 *
 * To help solve both of these problems I created the logging library.  This
 * module makes logging messages much easier while providing more helpful
 * information.  The library uses a #define macro to make the function calls
 * shorter.
 *
 * The #define macro in the program that is using the module:
 *
 \code
  #define LOGMESSAGE(string, location) logmessage(string, location,\
                   ALSAD_PROC_NAME, ALSAD_LOG_FILE, __FILE__,\
             __LINE__, __FUNCTION__);
 \endcode
 *
 * It is also necessary to #define ALSAD_PROC_NAME and ALSAD_LOG_FILE.
 *
 * These two variables are the name of the calling process and the name and
 * path of the log file the process should write to, respectively.
 *
 * A call such as:
 \code
 LOGMESSAGE("Hello World!", LOGSTDERR | LOGSYSLOG);
 \endcode
 *
 * logs and formats 'Hello World!' along with the other function parameters to
 * stderr and syslogd.
 *  
 *   
 */
 
/** \addtogroup logging_lib */
/** @{*/


#include "log_lib.h"


void logmessage(char *message, int unsigned location, char *process_name, 
	            char *log_file_name, char *filename, 
		      int line_num, char *function){
	        
   FILE* logdesc;
   struct timeval logtime; 
   char log_buffer[LOG_LIB_BUFF_SIZE];
   gettimeofday(&logtime, NULL);
 
   if((location & LOGSTDOUT) == LOGSTDOUT){
      fprintf(stdout,"%ld.%06ld Thread:%lu %s(%d): %s() :: %s\n\t %s\n",
			 logtime.tv_sec, logtime.tv_usec, pthread_self(),
			 filename, line_num, function, strerror(errno),
			 message);
   }
   
   if((location & LOGSTDERR) == LOGSTDERR){
      fprintf(stderr,"%ld.%06ld Thread:%lu %s(%d): %s() :: %s\n\t %s\n",
			 logtime.tv_sec, logtime.tv_usec, pthread_self(),
			 filename, line_num, function, strerror(errno),
			 message);
   }
   
   if((location & LOGFILE) == LOGFILE){
      logdesc = fopen(log_file_name, "a");
      if (logdesc == NULL) {
         fprintf(stderr, "logmessage: Could not open the log file\n");
      } 
      else {
         fprintf(logdesc,"%ld Thread:%lu %s(%d): %s() :: %s\n\t %s\n",
                         logtime.tv_usec, pthread_self(), filename,
			 line_num, function, strerror(errno), message);
         fclose(logdesc);
      }
   }
   
   if((location & LOGSYSLOG) == LOGSYSLOG){
      if((strlen(filename)+15+strlen(function)+strlen(message)+20)
		                                <LOG_LIB_BUFF_SIZE){
         sprintf(log_buffer,"%s(%d): %s() :: %s",
			 filename, line_num, function, message);
         openlog(process_name,LOG_PID,LOG_USER);
         syslog(LOG_INFO,"%s: %m",log_buffer);
      }
      else{
         fprintf(stderr,"logmessage: Log buffer not large enough.\n");
	 return;
      }
   }
   
   return;
}

/** @}*/
