/*! \file linked_list.c
 *  \brief This is the implementation of the Linked-List Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup linked_list_lib
*/


 /** \defgroup linked_list_lib The Linked-List Library
 *   \ingroup helper_components
 *   \brief Library that provides methods to add and remove elements to a
 *          thread-safe list.

**/
 
/** \addtogroup linked_list_lib */
/** @{*/



#include "linked_list.h"



static node_t *linked_list_init_node(void *data, node_t *prev,
                                        node_t *next);

static void *linked_list_destroy_node(node_t *node_to_destroy);

static void *linked_list_search_no_lock(linked_list_t *new_linked_list,
           int (generic_search_w_key)(),
           void *search_key);




static node_t *linked_list_init_node(void *data, node_t *prev, node_t *next){

   node_t *new_node;

   new_node=(node_t *)malloc(sizeof(node_t));
   if(new_node==NULL){
      fprintf(stderr,
         "linked_list_init_node: Could not allocate memory for new_node\n");
      return NULL;
   }
   new_node->data=data;
   new_node->prev=prev;
   new_node->next=next;

   return new_node;
}

static void *linked_list_destroy_node(node_t *node_to_destroy){

   void *return_ptr;

   if(node_to_destroy!=NULL)
      return_ptr=node_to_destroy->data;
   else
      return_ptr=NULL;

   free(node_to_destroy);

   return return_ptr;

}


int linked_list_init(linked_list_t **new_linked_list,
	                	size_t max_elements){


   (*new_linked_list)=(linked_list_t *)malloc(sizeof(linked_list_t));
   if(*new_linked_list==NULL){
      fprintf(stderr,
         "linked_list_init: Could not allocate memory for new_linked_list\n");
      return -1;
   }

   (*new_linked_list)->elements=0;

   (*new_linked_list)->max_elements=max_elements;

   (*new_linked_list)->head_of_list=NULL;

   pthread_mutex_init(&((*new_linked_list)->list_mutex),NULL);

   return 0;
}



int linked_list_destroy(linked_list_t *list_to_destroy){

   node_t *next_to_destroy;
   node_t *current;

   pthread_mutex_lock(&(list_to_destroy->list_mutex));

   if(list_to_destroy->elements>0){
      if(list_to_destroy->head_of_list==NULL){
         fprintf(stderr,
         "linked_list_destroy: Head of list is NULL ");
         fprintf(stderr,"but elements > 0, fatal error\n");
         return -1;
      }
      else{
         current=list_to_destroy->head_of_list;
         while(current!=NULL){
            next_to_destroy=list_to_destroy->head_of_list->next;
            free(current);
            current=next_to_destroy;
         }
      }  
   }

   pthread_mutex_destroy(&(list_to_destroy->list_mutex));

   free(list_to_destroy);

   return 0;
}


int linked_list_insert(linked_list_t *new_linked_list,
                   int (generic_search_w_key)(),
                   void *search_key, void *data_to_add){
   node_t *new_node;
   node_t *next_node;
   node_t *results;

   pthread_mutex_lock(&(new_linked_list->list_mutex));

   if((new_linked_list->elements)+1 > new_linked_list->max_elements &&
		   new_linked_list->max_elements!=0){
      fprintf(stderr,
      "linked_list_insert: Could not insert, max_elements reached.\n");
      pthread_mutex_unlock(&(new_linked_list->list_mutex));
      return -1;
   }

   results=(node_t *)linked_list_search_no_lock(new_linked_list,
                                      generic_search_w_key,
                                      search_key);
   if(results!=NULL){
      fprintf(stderr,
         "linked_list_insert: Node already exists.\n");
      pthread_mutex_unlock(&(new_linked_list->list_mutex));
      return -1;
   }

   new_node=linked_list_init_node(data_to_add,NULL,NULL);
   if(new_node==NULL){
      fprintf(stderr,
         "linked_list_insert: Could not initialize new node.\n");
      pthread_mutex_unlock(&(new_linked_list->list_mutex));
      return -1;
   }

   if(new_linked_list->head_of_list==NULL){
      new_linked_list->head_of_list=new_node;
   }
   else{
      next_node=new_linked_list->head_of_list;
      new_linked_list->head_of_list=new_node;
      new_node->next=next_node;
      next_node->prev=new_node;
   }

   new_linked_list->elements++;

   pthread_mutex_unlock(&(new_linked_list->list_mutex));

   return 0;
}


void *linked_list_remove(linked_list_t *new_linked_list, 
                         int (generic_search_w_key)(),
                         void *search_key){

   node_t *results;
   void *ret_ptr;

   pthread_mutex_lock(&(new_linked_list->list_mutex));


   results=(node_t *)linked_list_search_no_lock(new_linked_list,
                                      generic_search_w_key,
                                      search_key);
   if(results==NULL){
      pthread_mutex_unlock(&(new_linked_list->list_mutex));
      return NULL;
   }

   //only item in list
   if(results->prev==NULL && results->next==NULL){
      new_linked_list->head_of_list=NULL;
   }
   //head of list
   else if(results->prev==NULL && results->next!=NULL){
      results->next->prev=NULL;
      new_linked_list->head_of_list=results->next;
   }
   //tail of list
   else if(results->prev!=NULL && results->next==NULL){
      results->prev->next=NULL;
   }
   //somewhere in the middle
   else if(results->prev!=NULL && results->next!=NULL){
      results->prev->next=results->next;
      results->next->prev=results->prev;
   }
   
   ret_ptr=linked_list_destroy_node(results);

   new_linked_list->elements--;

   pthread_mutex_unlock(&(new_linked_list->list_mutex));

   return ret_ptr;

}




static void *linked_list_search_no_lock(linked_list_t *new_linked_list,
           int (generic_search_w_key)(),
           void *search_key){

   int retval;
   node_t *current;

   current=new_linked_list->head_of_list;
   while(current!=NULL){
      if((retval=generic_search_w_key(search_key, current->data))>=0){      
         return (void *)current;
      }
      current=current->next;
   }

   return NULL;

}



void *linked_list_search(linked_list_t *new_linked_list,
           int (generic_search_w_key)(),
           void *search_key){

   int retval;
   node_t *current;

   pthread_mutex_lock(&(new_linked_list->list_mutex));

   current=new_linked_list->head_of_list;
   while(current!=NULL){
      if((retval=generic_search_w_key(search_key, current->data))>=0){
            pthread_mutex_unlock(&(new_linked_list->list_mutex));
            return (void *)current->data;
      }
      current=current->next;
   }

   pthread_mutex_unlock(&(new_linked_list->list_mutex));

   return NULL;

}


void *linked_list_rand_search(linked_list_t *new_linked_list,
                                    int (generic_search_wo_key)()){
   return NULL;

}






int linked_list_process_node(linked_list_t *new_linked_list, 
                             int (generic_search_w_key)(),
                             void *search_key,
                             int (generic_processing)(),
                             void *args){ 
                                
   node_t *current;
   int retval;

   pthread_mutex_lock(&(new_linked_list->list_mutex));
   
   if(new_linked_list->elements == 0){
      pthread_mutex_unlock(&(new_linked_list->list_mutex));
      return 0;
   }
      
   current=(node_t *)linked_list_search_no_lock(new_linked_list,
                                      generic_search_w_key,
                                      search_key);
   if(current==NULL){
      pthread_mutex_unlock(&(new_linked_list->list_mutex));
      return -1;
   }
   
   if((retval=generic_processing(current->data, args))<0){
      pthread_mutex_unlock(&(new_linked_list->list_mutex));
      return -1;
   }

   pthread_mutex_unlock(&(new_linked_list->list_mutex));

   return 0; 

}



int linked_list_process_all(linked_list_t *new_linked_list,
	                                     int (generic_processing)(),
                                        void *args){
   
   node_t *current;
   int retval;

   pthread_mutex_lock(&(new_linked_list->list_mutex));
   
   if(new_linked_list->elements == 0){
      pthread_mutex_unlock(&(new_linked_list->list_mutex));
      return 0;
   }
   
   current=new_linked_list->head_of_list;
   while(current!=NULL){
      if((retval=generic_processing(current->data, args))<0){
         pthread_mutex_unlock(&(new_linked_list->list_mutex));
         return -1;
      }
      current=current->next;
   }
   

   pthread_mutex_unlock(&(new_linked_list->list_mutex));

   return 0; 

}






void *linked_list_compare_all(linked_list_t *new_linked_list,
                                        int (generic_compare_nodes)(),
                                        void *args){
   
   node_t *current;
   void *winning_data;
   int retval;

   winning_data=NULL;
   
   pthread_mutex_lock(&(new_linked_list->list_mutex));
   
   if(new_linked_list->elements == 0){
      pthread_mutex_unlock(&(new_linked_list->list_mutex));
      return NULL;
   }
   
   current=new_linked_list->head_of_list;
   if(current!=NULL){
      winning_data=current->data;
      current=current->next;
   }
   while(current!=NULL){
      if((retval=generic_compare_nodes(current->data, winning_data, args))==0){
         winning_data=current->data;
      }
      current=current->next;
   }
   
   pthread_mutex_unlock(&(new_linked_list->list_mutex));

   return winning_data; 

}

/** @}*/
