#ifndef __CONFIG_FILE_H
#define __CONFIG_FILE_H

 
/** \addtogroup config_file_lib */
/** @{*/
 
/*! \file config_file.h
*   \brief This is the header for the Configuration-File Library
*   \author Daniel R. Warren
*   \version 1.0
*   \date    November 2004
*/


#define _GNU_SOURCE                 //For getline function in stdio.h

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#define CONFIG_FILE_MAX_PAIRS 64 
#define CONFIG_FILE_MAX_RESIZE 32



/** \brief Storage structure for a key value pair */
struct key_value_pair{
  char *key;             /**< Pointer to the key pulled from the config file */
  char *value;           /**< Pointer to the value for the corresponding key */
};
typedef struct key_value_pair key_value_pair_t;



/** \brief Reads the requested filename and returns the 
 *         key-value pairs from within
 * 
 * This function opens filename and allocates memory for CONFIG_FILE_MAX_PAIRS
 * key_value_pair_t.  The function then iterates through the file pulling
 * key-value pairs into the array of memory allocated for key_value_pair_t.
 * The buffer is resized if CONFIG_FILE_MAX_PAIRS is exceded.  
 * CONFIG_FILE_MAX_RESIZE * CONFIG_FILE_MAX_PAIRS is equal to the maximum number
 * of key_value_pair_t that can be returned with this function call.
 * 
 * @param filename This is a character pointer that points to the full path of
 *        the configuration file.
 * @param pairs_loaded This returns the number of key_value_pair_t that were
 *        loaded.
 * @see config_file_write()
 * @return This function returns a pointer to a list of key_value_pair_t.  These
 *        are the key-value pairs that are pulled from filename.
 * 
**/
key_value_pair_t *config_file_read(char *filename, int *pairs_loaded);


/** \brief Writes the key-value pairs to filename
 * 
 * This function opens filename and iterates through pairs writing each
 * element to the file in the format:
 * \verbatim
   <key>=<value>
   \endverbatim
 * 
 * @param filename A character pointer that points to the full path of
 *        the configuration file.
 * @param pairs A pointer to a list of key_value_pair_t to write to filename
 * @param pairs_to_write The number of key_value_pair_t in pairs.
 * @see config_file_read()
 * @return returns 0 on success and -1 on failure.
 * 
**/
int config_file_write(char *filename, key_value_pair_t *pairs,
	                                	int pairs_to_write);



/** \brief Searches pairs for key and returns the value of that key.
 * 
 * Iterates through pairs comparing key to the key-value pairs.  If a match is
 * found the function returns the value associated with that key.
 * 
 * @param pairs An array of key_value_pair_t
 * @param key A pointer to the key to be searched for in pairs
 * @param pairs_loaded The number of key_value_pair_t in pairs
 * @return This function returns a pointer to the value associated with key if
 *         key is found in pairs.  If key is not found in pairs, NULL is
 *         returned.
**/
char *config_file_get_value(key_value_pair_t *pairs, char *key,
	                                 	int pairs_loaded);

/**\todo Create config_file_set_value to update a value in an array of
 * key_value_pair_t.   */

/** \brief This function frees all of the memory that has been allocated with
 *         config_file_read().
 * 
 * This function iterates through the list of key-value pairs and frees the
 * keys and values.  The memory allocated for the array of key_value_pair_t
 * is also freed.
 * @param pairs An array of key_value_pair_t
 * @param pairs_loaded The number of key_value_pair_t in pairs.
 * @see config_file_read()
 * @return This function returns a pointer to a list of key_value_pair_t.  These
 *        are the key-value pairs that are pulled from the specified file.
 * 
**/
void config_file_destroy(key_value_pair_t *pairs, int pairs_loaded);

/** @}*/

#endif
