#ifndef __CIRC_BUFF_H
#define __CIRC_BUFF_H

/** \addtogroup circ_buff_lib */
/** @{*/
 
/*! \file circ_buff.h
*   \brief This is the header for the Circular-Buffer Library
*   \author Daniel R. Warren
*   \version 1.0
*   \date    November 2004
*/


#define _GNU_SOURCE
#include <string.h>
#include <stdlib.h>
#include <stdio.h> 
#include <sys/time.h>
#include <errno.h>
#include <pthread.h>



/*
This is essentially just a struct that has a couple of accessor functions
to use on it.  The struct contains a pointer to the address of the buffer
and other essential items like the size of the buffer the offset of both
the read point and the write point.  The read point is defined as the 
starting point with which to read from the buffer.  The write point is the
location to start writing when data gets put into the buffer.  Another
item that is in the struct is how much data is in the buffer.  The struct
*/

/** \brief Storage structure for a key value pair */
struct circ_buff {
   long buff_size;       /**< Size of the buffer in bytes */
   long write_pnt;       /**< Starting point to write */
   long read_pnt;        /**< Starting point to read  */
   long byte_count;      /**< Number of bytes currently in the buffer */
   unsigned long long byte_id;         /**< Identifier for each byte in the buffer */
   pthread_mutex_t circ_write_mut; /**< Mutex to prevent simultaneous writes */
   pthread_mutex_t circ_read_mut;   /**< Mutex to prevent simultaneous reads */
   pthread_mutex_t circ_size_var_mut; /**< Mutex to prevent simultaneous
                                           updates of the size variable */
   pthread_mutex_t circ_write_cond_mut; /**< Mutex for use with 
                                           circ_write_cond_var to wait for a
                                           write */
   pthread_cond_t circ_write_cond_var;  /**< Condition variable that allows 
                                           threads to know when a write has been
                                           made */
   pthread_mutex_t circ_read_cond_mut;  /**< Mutex for use with 
                                             circ_read_cond_var to wait for a
                                             read*/
   pthread_cond_t circ_read_cond_var;  /**< Condition variable that allows
                                            threads to know when a read has been
                                            made */
   void *buff_ptr;     /**< Pointer to the memory location of the buffer */
};
 
typedef struct circ_buff circ_buff_t;


int circ_buff_init(circ_buff_t **new_buff, long buff_size);

void circ_buff_destroy(circ_buff_t *new_buff);

int circ_buff_init_thread(circ_buff_t *new_buff, long *thread_read_pnt,
	                        	unsigned long long *thread_byte_id,
                                struct timespec *timeout);

int circ_buff_wait_to_start(circ_buff_t *new_buff,
	                                 unsigned long long thread_byte_id,
                                     struct timespec *timeout);

int circ_buff_wait_for_read(circ_buff_t *new_buff, struct timespec *timeout);

int circ_buff_wait_for_write(circ_buff_t *new_buff, struct timespec *timeout);

int circ_buff_resize(circ_buff_t *new_buff, long new_size,
                                            struct timespec *timeout);

long circ_buff_read(circ_buff_t *new_buff, void *read_data,
	                                	long bytes_to_read,
                                        struct timespec *timeout);

long circ_buff_read_thread(circ_buff_t *new_buff, void *read_data,
                           long *thread_read_pnt,
                           unsigned long long *thread_byte_id,
				           long read_size,
                           struct timespec *timeout);

long circ_buff_write(circ_buff_t *new_buff, void * write_data,
	                   	long bytes_to_write,
                        struct timespec *timeout);



#endif
