/*! \file auth_lib.c
 *  \brief This is the implementation of the Authentication Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup authentication_lib
*/



 /** \defgroup authentication_lib The Authentication Library
 *   \ingroup helper_components
 *   \brief This library creates a simple interface for server processes to
 *          authenticate clients by using public/private key encryption.
 * 
 * 
 * The process of creating a daemon requires a developer to make many difficult
 * design decisions.  One of these decisions is the choice of authentication
 * methods.  Most daemons are required to use authentication because they
 * provide data that should only be accessible to select users.  There are two
 * major categories of authentication, symmetric and asymmetric.  Symmetric
 * authentication bases authentication on a secret shared between client and
 * server.  Suppling a password to an FTP server is an example of this.  Both
 * the server and the client have to keep track of the client's password, which
 * is the secret.  Asymmetric authentication allows the server to authenticate
 * the client without sharing secrets.  This method is more secure because
 * there is no secret passing over lines of communication.  Public/private key
 * authentication uses this method.
 *
 * Public/private key encryption is based on the principle that data encrypted
 * with a client's public key can only be decrypted (in a reasonable amount
 * of time) by that client's private key, and vice versa.  This process also
 * assumes that the private key is known by the client only.
 *
 * In order to send a private message to a server, the client should encrypt
 * the message with the server's public key, and then the server decrypts it
 * with its private key.
 *
 * To authenticate a client, a server sends it a challenge: a string of random
 * bytes.  The client then encrypts the challenge using its private key and
 * sends the result back to the server.  The server uses the client's public
 * key to decrypt the message.  If the resulting message matches the challenge;
 * the client is authenticated.
 *
 * Alsad provides an implementation of the process described above in the
 * authentication library.  The library is divided into two main functions, a
 * server function and a client function.  These functions are called by the
 * client and server processes after a TCP connection has been established.

 * For the two functions of this library to work properly there is some initial
 * setup that must occur.  First, any client that the server needs to
 * authenticate must have a public key file in pub_key_dir on the server that
 * corresponds to the user_name supplied.  The public key file on the server
 * should be in the format: 
 
 \verbatim
  <user_name>.pub
 \endverbatim
 
 * The naming convention for priv_key_file is not important because the
 * variable specifies the full path.  The public and private keys used in this
 * library are read using the OpenSSL libraries.  These libraries must be in
 * place for the authentication library to compile.  The library also expects
 * the use of PEM format keys with no pass phrase.  These keys are generated by
 * typing the following commands at the shell:
 
 \code
  openssl genrsa -out output_file 1024
 \endcode
 
 * This command creates a private key file output_file in PEM format that does 
 * not require a pass phrase and has a 1024-bit modulus.
 *
 * In order to create the public key in PEM format, use the following command
 * at the shell:
 \code
  openssl rsa -in private_key_file -out output_file -pubout
 \endcode
 
 * This command will create the public key file output_file in PEM format from
 * the private key found in private_key_file.
 * 
 * 
**/


/** \addtogroup authentication_lib */
/** @{*/




#include "auth_lib.h"

static int auth_send_plain_text(int sock, char *plain_text, 
                                                unsigned int datalength);

static int auth_recv_plain_text(int sock, char **plain_text,
                                                unsigned int *datalength);

static int auth_encrypt_and_send_msg(int sock, char *decrypt_msg, int size,
      RSA *rsa_priv_key);

static int auth_recv_and_decrypt_msg(int sock, char **decrypt_msg, int *size,
        RSA *rsa_pub_key);

int auth_server(int sock, char *pub_key_dir){

   int ret_val;
   ssize_t sret_val;
   char *user_name;
   int user_name_len;
   char *usr_pub_key_file;
   int total_name_len;
   size_t rand_length;
   char *rand_bytes;
   char *decrypt_msg;
   int decrypt_size;
   char *pub_key_file_ext=AUTH_PUB_KEY_FILE_EXT;
   FILE *public_key_file;
   int dev_random_handle;
   RSA *rsa_pub_key;


   ret_val=auth_recv_plain_text(sock,&user_name,&user_name_len);
   if(ret_val < 0){
      fprintf(stderr,"auth_server: Could not recv user_name\n");
      return -1;
   }


   total_name_len=strlen(pub_key_dir)+strlen(user_name)+
	   strlen(pub_key_file_ext)+1;
   usr_pub_key_file=(char *)malloc(total_name_len);
   if(usr_pub_key_file==NULL){
      fprintf(stderr,"auth_server: Could not malloc file name\n");
      free(user_name);
      return -1;
   }

   memcpy(usr_pub_key_file,pub_key_dir,strlen(pub_key_dir)+1);
   strcat(usr_pub_key_file,user_name);
   strcat(usr_pub_key_file,pub_key_file_ext);

   free(user_name);

   public_key_file=fopen(usr_pub_key_file,"r");
   free(usr_pub_key_file);
   if(public_key_file==NULL){
      fprintf(stderr,"auth_server: Could not open public key file\n");
      ret_val=auth_send_plain_text(sock,AUTH_INVALID_MESSAGE,
		              strlen(AUTH_INVALID_MESSAGE)+1);
      if(ret_val < 0){
         fprintf(stderr,"auth_server: Could not send invalid message\n");
      }
      return -1;
   }
   rsa_pub_key=NULL;
   rsa_pub_key=PEM_read_RSA_PUBKEY(public_key_file, NULL, NULL, NULL);
   if(rsa_pub_key==NULL){
       fprintf(stderr,"auth_server: Could not read public key\n");
       //ERR_load_crypto_strings();
       //error_code=ERR_get_error();
       //perror(ERR_error_string(error_code,NULL));
       return -1;
   }

   fclose(public_key_file);
   
   rand_length=1+(int) (AUTH_MAX_ENCR_MESS_LEN*rand()/(RAND_MAX+1.0));

   rand_bytes=(char *)malloc(sizeof(char)*rand_length);
   if(rand_bytes==NULL){
      fprintf(stderr,"auth_server: Could not malloc random bytes\n");
      RSA_free(rsa_pub_key);
      return -1;
   }

   //now using /dev/random
   
   dev_random_handle=open("/dev/random",O_RDONLY);
   if(dev_random_handle<0){
      fprintf(stderr,"auth_server: Could not open /dev/random\n");
      RSA_free(rsa_pub_key);
      return -1;
   }

   sret_val=read(dev_random_handle, rand_bytes, rand_length);
   if(sret_val<0){
      fprintf(stderr,"auth_server: Unable to read from /dev/random\n");
      RSA_free(rsa_pub_key);
      return -1;
   } 
   
  // for(i=0;i<rand_length;i++){
  //    rand_temp=rand();
  //    memcpy(rand_bytes+(i*sizeof(int)),&rand_temp,sizeof(int));
  // }


   ret_val=auth_send_plain_text(sock,rand_bytes,rand_length*sizeof(char));
   if(ret_val < 0){
      fprintf(stderr,"auth_server: Could not send random bytes\n");
      free(rand_bytes);
      RSA_free(rsa_pub_key);
      return -1;
   }

   ret_val=auth_recv_and_decrypt_msg(sock,&decrypt_msg,&decrypt_size,
		                    rsa_pub_key);

   RSA_free(rsa_pub_key);
   if(ret_val < 0){
      fprintf(stderr,"auth_server: Could not recv encrypted message\n");
      free(rand_bytes);
      return -1;
   }

   /* Prints out after decryption
   if(sizeof(int)*rand_length==decrypt_size){
      for(i=0;i<decrypt_size;i++){
         fprintf(stderr,"rand_bytes= %x,\tdecrypt_msg=%x\n",
		      rand_bytes[i],*((*decrypt_msg)+i));
      }
   }
   */

   if(sizeof(char)*rand_length==decrypt_size &&
		   0==memcmp(rand_bytes,decrypt_msg,decrypt_size)){
      free(rand_bytes);
      free(decrypt_msg);
      ret_val=auth_send_plain_text(sock, AUTH_VALID_MESSAGE,
		                 strlen(AUTH_VALID_MESSAGE)+1);
      if(ret_val < 0){
         fprintf(stderr,"auth_server: Could not send validation message\n");
         return -1;
      }
      return 0;
   }

   free(rand_bytes);
   free(decrypt_msg);
   ret_val=auth_send_plain_text(sock, AUTH_INVALID_MESSAGE,
		              strlen(AUTH_INVALID_MESSAGE)+1);
   if(ret_val < 0){
      fprintf(stderr,"auth_server: Could not send invalid message\n");
   }

   return -1;
}



int auth_client(int sock, char *user_name, char *usr_priv_key){

   int ret_val;
   int rand_len;
   int result_len;
   char *result_msg;
   char *rand_bytes;
   FILE *private_key_file;
   RSA *rsa_priv_key;

   private_key_file=fopen(usr_priv_key,"r");
   if(private_key_file==NULL){
      fprintf(stderr,"auth_client: Could not open private key file\n");
      return -1;
   }
   rsa_priv_key=PEM_read_RSAPrivateKey(private_key_file, NULL,
		                                             NULL, NULL);
   fclose(private_key_file);
   if(rsa_priv_key==NULL){
       fprintf(stderr,"auth_client: Could not read private key\n");
       RSA_free(rsa_priv_key);
       return -1;
   }

   ret_val=auth_send_plain_text(sock,user_name,strlen(user_name)+1);
   if(ret_val < 0){
      fprintf(stderr,"auth_client: Could not send user name\n");
      RSA_free(rsa_priv_key);
      return -1;
   }

   ret_val=auth_recv_plain_text(sock, &rand_bytes, &rand_len);
   if(ret_val < 0){
      fprintf(stderr,"auth_client: Could not receive random bytes\n");
      RSA_free(rsa_priv_key);
      return -1;
   }

   if(0==strcmp(rand_bytes,AUTH_INVALID_MESSAGE)){
      fprintf(stderr, "auth_client: No public key file on server\n");
      RSA_free(rsa_priv_key);
      free(rand_bytes);
      return -1;
   }

   ret_val=auth_encrypt_and_send_msg(sock, rand_bytes, rand_len,
		rsa_priv_key);

   free(rand_bytes);
   RSA_free(rsa_priv_key);
   if(ret_val < 0){
      fprintf(stderr,"auth_client: Could not send decrypted message\n");
      return -1;
   }

   ret_val=auth_recv_plain_text(sock, &result_msg, &result_len);
   if(ret_val < 0){
      fprintf(stderr,"auth_client: Could not receive result message\n");
      return -1;
   }

   if(0==strncmp(result_msg,AUTH_VALID_MESSAGE,strlen(AUTH_VALID_MESSAGE))){
      free(result_msg);
      return 0;
   }

   fprintf(stderr,"auth_client: server did not validate message\n");
   free(result_msg);
   return -1;

}


static int auth_send_plain_text(int sock, char *plain_text,
                                                unsigned int datalength){

   unsigned int packetlength;
   int ret_val;

   char *packet_buff;

   packetlength=datalength+sizeof(unsigned int);

   packet_buff=(char *)malloc(packetlength);
   if(packet_buff==NULL){
      fprintf(stderr,"auth_send_plain_text: failed to malloc packet buffer\n");
      return -1;
   }

   datalength=htonl(datalength);
   memcpy(packet_buff,&datalength,sizeof(unsigned int));
   datalength=ntohl(datalength);
   memcpy(packet_buff+sizeof(unsigned int),plain_text,datalength);

   ret_val=safe_sock_send(sock,packet_buff,packetlength, AUTH_SOCK_TIMEOUT);
   free(packet_buff);
   if(ret_val!=packetlength){
      fprintf(stderr,"auth_send_plain_text: Could not send packet\n");
      return -1;
   }
   return 0;
}



static int auth_recv_plain_text(int sock, char **plain_text,
                                                 unsigned int *datalength){

   int ret_val;

   ret_val=safe_sock_recv(sock,(char *)datalength,sizeof(unsigned int), 
                                                  AUTH_SOCK_TIMEOUT);
   if(ret_val!=sizeof(unsigned int)){
      fprintf(stderr,"auth_recv_plain_text: Could not recv size of packet\n");
      return -1;
   }

   *datalength=ntohl(*datalength);

   if(*datalength>AUTH_MAX_TEXT_LEN){
      fprintf(stderr,"auth_recv_plain_text: ");
      fprintf(stderr,"length of text greater than max\n");
      return -1;
   }

   *plain_text=(char *)malloc(*datalength);
   if(*plain_text==NULL){
      fprintf(stderr,"auth_recv_plain_text: ");
      fprintf(stderr,"could not allocate memory for text buff\n");
      return -1;
   }

   ret_val=safe_sock_recv(sock,*plain_text,*datalength, AUTH_SOCK_TIMEOUT);
   if(ret_val!=*datalength){
      fprintf(stderr,"auth_send_plain_text: Could not recv data packet\n");
      free(*plain_text);
      return -1;
   }

   return 0;
}







static int auth_encrypt_and_send_msg(int sock, char *decrypt_msg, int size,
		                                             RSA *rsa_priv_key){

   unsigned int packetlength;
   int ret_val;

   char *packet_buff;
   char *encrypt_msg;

   encrypt_msg=(char *)malloc(RSA_size(rsa_priv_key));
   if(encrypt_msg==NULL){
      fprintf(stderr,"auth_encrypt_and_send_msg: ");
      fprintf(stderr,"failed to malloc encrypt buffer\n");
      return -1;
   }
   size=RSA_private_encrypt(size, decrypt_msg, encrypt_msg, rsa_priv_key,
		              RSA_PKCS1_PADDING);
   if(size<0){
      fprintf(stderr,"auth_encrypt_and_send_msg: could not encrypt message\n");
      free(encrypt_msg);
      return -1;
   }

   packetlength=size+sizeof(unsigned int);

   packet_buff=(char *)malloc(packetlength);
   if(packet_buff==NULL){
      fprintf(stderr,"auth_encrypt_and_send_msg: ");
      fprintf(stderr,"failed to malloc packet buffer\n");
      free(encrypt_msg);
      return -1;
   }
   size=htonl(size);
   memcpy(packet_buff,&size,sizeof(unsigned int));
   size=ntohl(size);
   memcpy(packet_buff+sizeof(unsigned int),encrypt_msg,size);

   ret_val=safe_sock_send(sock, packet_buff, packetlength, AUTH_SOCK_TIMEOUT);
   free(packet_buff);
   free(encrypt_msg);
   if(ret_val!=packetlength){
      fprintf(stderr,"auth_encrypt_and_send_msg: Could not send packet\n");
      return -1;
   }

   return 0;
}







static int auth_recv_and_decrypt_msg(int sock, char **decrypt_msg, int *size,
		                                              RSA *rsa_pub_key){

   int ret_val;
   char *encrypt_msg;

   ret_val=safe_sock_recv(sock,size,sizeof(unsigned int), AUTH_SOCK_TIMEOUT);
   if(ret_val!=sizeof(unsigned int)){
      fprintf(stderr,"auth_recv_and_decrypt_msg: ");
      fprintf(stderr,"Could not recv size of packet\n");
      return -1;
   }

   *size=ntohl(*size);

   if(*size>AUTH_MAX_MSG_LEN){
      fprintf(stderr,"auth_recv_and_decrypt_msg: ");
      fprintf(stderr,"length of message greater than max\n");
      return -1;
   }

   encrypt_msg=(char *)malloc(*size);
   if(encrypt_msg==NULL){
      fprintf(stderr,"auth_recv_and_decrypt_msg: could not allocate");
      fprintf(stderr," memory for encrypt buff\n");
      return -1;
   }

   ret_val=safe_sock_recv(sock,encrypt_msg,*size, AUTH_SOCK_TIMEOUT);
   if(ret_val!=*size){
      fprintf(stderr,"auth_send_plain_text: Could not recv data packet\n");
      free(encrypt_msg);
      return -1;
   }


   *decrypt_msg=(char *)malloc(RSA_size(rsa_pub_key) - 11);
   if(decrypt_msg==NULL){
      fprintf(stderr,"auth_recv_and_decrypt_msg: could not allocate");
      fprintf(stderr," memory for msg buff\n");
      free(encrypt_msg);
      return -1;
   }

   *size=RSA_public_decrypt(*size, encrypt_msg, *decrypt_msg, rsa_pub_key,
		              RSA_PKCS1_PADDING);

   free(encrypt_msg);
   if(*size<0){
      free(*decrypt_msg);
      return -1;
   }

   return 0;
}

/** @}*/
