/*! \file alsad_stream_list.c
 *  \brief This is the implementation of the Stream-List Library
 *  \author Daniel R. Warren
 *  \version 1.0
 *  \date    November 2004
 *  \ingroup stream_list
*/


 /** \defgroup stream_list The Stream-List Library
 *   \ingroup application_components
 *   \brief Library that provides a mechanism for storing stream list nodes.
 * 
   
 */
 
/** \addtogroup stream_list */
/** @{*/


#include "alsad_stream_list.h"


int alsad_stream_list_init(alsad_stream_list_t *new_list,
                                               unsigned int num_streams){
   int retval;
  
   if((retval=linked_list_init(&(new_list->streams),num_streams)) < 0){
      fprintf(stderr,"alsad_stream_list_init: ");
      fprintf(stderr,"linked_list_init failed for sources\n");
      return -1;
   }
   
   new_list->shutdown=0;

   return 0;
}


void alsad_stream_list_destroy(alsad_stream_list_t *destroy_list){
   /** \todo Create a special list destory function that waits for the list
    * to empty.
    * 
    */
   linked_list_destroy(destroy_list->streams);

}


int alsad_stream_node_init(alsad_stream_node_t **new_stream_node, 
		                      alsad_stream_props_t *new_stream,
		                      alsad_config_keys_t *keys){
	
   unsigned long buffer_size_in_bytes;
   unsigned int max_circ_buff_size;
   unsigned int max_stream_sinks;
   int retval;
   alsad_stream_props_t *stored_stream_props;
   
   
   *new_stream_node=(alsad_stream_node_t *)malloc(sizeof(alsad_stream_node_t));
   if(new_stream_node==NULL){
      fprintf(stderr,"Could not allocate memory for new_stream_node,");
      return -1; 
   }
   
   (*new_stream_node)->closing=0;
      
   stored_stream_props=(alsad_stream_props_t *)malloc(sizeof(alsad_stream_props_t));
   if(stored_stream_props==NULL){
      fprintf(stderr,"Could not allocate memory for rcvd_local_stream");
      free(new_stream_node);
      return -1; 
   }
  
   memcpy(stored_stream_props, new_stream, sizeof(alsad_stream_props_t));
   
   (*new_stream_node)->stream_props=stored_stream_props;

   max_circ_buff_size=atoi(keys->config_max_circ_buff_size);  

   if(stored_stream_props->circ_buff_size > max_circ_buff_size){
      buffer_size_in_bytes=max_circ_buff_size;
      stored_stream_props->circ_buff_size=max_circ_buff_size;
   }
   
   buffer_size_in_bytes=stored_stream_props->circ_buff_size;

   if(buffer_size_in_bytes < stored_stream_props->frame_size * 
                                           stored_stream_props->rate){
      buffer_size_in_bytes=stored_stream_props->frame_size *
                           stored_stream_props->rate;
      stored_stream_props->circ_buff_size=buffer_size_in_bytes;
      
   }
   fprintf(stderr,"alsad_stream_node_init: ");
   fprintf(stderr,"Buffer size used; %lu\n", buffer_size_in_bytes);
   retval=circ_buff_init(&(*new_stream_node)->audio_circ_buff, 
		          buffer_size_in_bytes);
   if(retval<0){
      fprintf(stderr,"alsad_stream_node_init: ");
      fprintf(stderr,"Create failed could not init_circ_buff\n");
      free(new_stream_node);
      free(stored_stream_props);
      return -1;
   }

   //Change 1 to be a new key in the config keys function
   if((retval=alsad_data_pipe_list_init(&(*new_stream_node)->sources,1)) < 0){
      fprintf(stderr,"alsad_stream_node_init: ");
      fprintf(stderr,"linked_list_init failed for sources\n");
      circ_buff_destroy((*new_stream_node)->audio_circ_buff);
      
      free(new_stream_node);
      free(stored_stream_props);
      return -1;
   }

   max_stream_sinks=atoi(keys->config_max_stream_sinks);  
   
   if((retval=alsad_data_pipe_list_init(&(*new_stream_node)->sinks,
                                                    max_stream_sinks)) < 0){
      fprintf(stderr,"alsad_stream_node_init: ");
      fprintf(stderr,"linked_list_init failed for sources\n");
      alsad_data_pipe_list_destroy(&(*new_stream_node)->sources);
      circ_buff_destroy((*new_stream_node)->audio_circ_buff);
      free(stored_stream_props);
      free(new_stream_node);

      return -1;
   }

   return 0;
}




/**TODO This function should look at the sources and sinks linked list and
 * destroy the lists only when they are empty
 */


void alsad_stream_node_destroy(alsad_stream_node_t *destroy_stream){

   circ_buff_destroy(destroy_stream->audio_circ_buff);
   alsad_data_pipe_list_destroy(&destroy_stream->sources);
   alsad_data_pipe_list_destroy(&destroy_stream->sinks);
   free(destroy_stream->stream_props);
   free(destroy_stream);
}




alsad_stream_node_t *alsad_stream_list_insert(
		                     alsad_stream_list_t *list,
	                             alsad_stream_node_t *new_stream_node){
   int retval;
   
   if((retval=linked_list_insert(list->streams,
                            alsad_stream_node_compare,
                            (void *)&(new_stream_node->stream_props->identifier),
                            (void *)new_stream_node)) < 0){
      fprintf(stderr,"Could not insert stream into linked list\n");
      return NULL;
   }

   return new_stream_node;
   
}









alsad_stream_node_t* alsad_stream_list_remove(
		               alsad_stream_list_t *list,
			       unsigned int ident){

   alsad_stream_node_t *retptr;

   if((retptr=(alsad_stream_node_t *)linked_list_remove(list->streams,
                            alsad_stream_node_compare,
                            (void *)&ident)) == NULL){
      fprintf(stderr,"Could not remove stream from linked list\n");
   }

   return retptr;


}



alsad_stream_node_t *alsad_stream_list_find(alsad_stream_list_t *list,
	        	unsigned int ident){

   alsad_stream_node_t *retptr;

   if((retptr=(alsad_stream_node_t *)linked_list_search(list->streams,
                            alsad_stream_node_compare,
                            (void *)&ident)) == NULL){
      fprintf(stderr,"Could not find stream in linked list\n");
   }

   return retptr;

  
}

//Not yet implemented. I'm not sure there is a need.
/*
void alsad_print_stream_list(alsad_stream_list_t *list){

   alsad_stream_node_t *current;
   
   pthread_mutex_lock(&(list->list_mutex)); 
   
   if(list->first_element==NULL){
      pthread_mutex_unlock(&(list->list_mutex)); 
      return;
   }

   current = list->first_element;


   fprintf(stdout,"\nPrinting list....\n");
   if(current == NULL){
      fprintf(stdout,"(empty)");
      return;
   }

   while(current != NULL){

      fprintf(stdout,"\nIdentifier: %d\n", current->audio_chan->identifier);
      fprintf(stdout,"Text Description: %s\n", current->audio_chan->text_desc);
      current = current->next_chan;

   }

   pthread_mutex_unlock(&(list->list_mutex)); 
   return;

}

*/


int alsad_stream_list_send(int sock, alsad_stream_list_t *list){

   int retval;
   alsad_send_stream_list_args_t list_args;
   
   if(list==NULL)
      return -1;

   list_args.sock=sock;
       
   if((retval=linked_list_process_all(list->streams,
                                      alsad_stream_send_stream_nodes,
                                      (void *)&list_args)) < 0){ 
      fprintf(stderr,"Could not send all streams\n");
      return -1;
   }


   return 0;

}


int alsad_stream_list_send_pipes(int sock, alsad_stream_list_t *list,
                                                   unsigned int ident){

   int retval;
   alsad_stream_node_t *return_stream_node;
   alsad_control_t send_control;
   
   return_stream_node=alsad_stream_list_find(list, ident);
   if(return_stream_node==NULL){
      fprintf(stderr,"Could not find stream to send to client\n");
      return -1;
   }
   
   send_control.ctrl=ALSAD_MSG_RQST_OK;
   if((retval=alsad_send_control(sock, &send_control)) <= 0){
      fprintf(stderr,"Failed to send control message to client\n");
      return -1;
   }
   
   if((retval=alsad_data_pipe_list_send(sock,
                                  &return_stream_node->sources, 0)) < 0){
      fprintf(stderr,"Could not send source pipes to client\n");
      return -1;
   }
   
   
   if((retval=alsad_data_pipe_list_send(sock,
                                    &return_stream_node->sinks, 1)) < 0){
      fprintf(stderr,"Could not send sink pipes to client\n");
      return -1;
   }
   
   
   return 0;   

}



int alsad_stream_node_compare(void *identifier, void *stream_node){

   unsigned int *tmp_identifier;
   alsad_stream_node_t *tmp_stream_node;

   tmp_identifier=(unsigned int *)identifier; 
   tmp_stream_node=(alsad_stream_node_t *)stream_node; 

   if((*tmp_identifier)==tmp_stream_node->stream_props->identifier)
      return 0;
    
   return -1;

}



int alsad_stream_send_stream_nodes(void *stream_node, void *args){

   int retval;
   
   if((retval=alsad_send_stream_props(((alsad_send_stream_list_args_t *)args)->sock,
                     ((alsad_stream_node_t *)stream_node)->stream_props)) <= 0){
      fprintf(stderr,"Could not send stream to client\n");
      return -1;
   }
   
   return 0;   

}



/** @}*/


