#ifndef __ALSAD_NET_LIB_H
#define __ALSAD_NET_LIB_H

/** \addtogroup net_lib */
/** @{*/
 
/*! \file alsad_net_lib.h
*   \brief This is the header for the Communications Library
*   \author Daniel R. Warren
*   \version 1.0
*   \date    November 2004
*/

#include "alsad_defs.h"
#include <netinet/in.h>
#include <alsa/asoundlib.h>
#include "socket_lib.h"


#define ALSAD_MAX_DEV_LENGTH 40
#define ALSAD_MAX_TEXT_DESC  65
#define ALSAD_MAX_HOST_LEN   257

/*
   Control structure for alsad network protocol
*/
struct alsad_control{
  unsigned int ctrl;
  unsigned int code;
}; 

typedef struct alsad_control alsad_control_t;


/*
   Structure for passing hardware parameters over the network
*/
struct alsad_hw_params{
   snd_pcm_stream_t stream;
   int open_mode;
   unsigned int buffer_time;
   unsigned int period_time;
   unsigned long buffer_size;
   unsigned long period_size;
   char pcm_name[ALSAD_MAX_DEV_LENGTH];		// name of pcm device to open
};

typedef struct alsad_hw_params alsad_hw_params_t;


/*
   Structure for passing software parameters over the network
*/
struct alsad_sw_params{
   int foo;
   int bar;
};

typedef struct alsad_sw_params alsad_sw_params_t;


/*
   Audio data header
*/
struct alsad_audio_hdr{
   unsigned int payload_size;
};

typedef struct alsad_audio_hdr alsad_audio_hdr_t;


/*
 * Channel header that contains properties of a channel
 */
struct alsad_stream_props{
   unsigned int identifier;
   snd_pcm_format_t format;
   unsigned int channels;
   unsigned int rate;
   unsigned int frame_size;
   unsigned int circ_buff_size;
   unsigned int src_permission;
   char text_desc[ALSAD_MAX_TEXT_DESC];
};
typedef struct alsad_stream_props alsad_stream_props_t;


/*
 * Address information of connected client.
 */
struct alsad_connect_addr{
   int port;
   char hostname[ALSAD_MAX_HOST_LEN];
};

typedef struct alsad_connect_addr alsad_connect_addr_t;



/*
 * Stored information for sources and sinks.
 */
struct alsad_data_pipe{
   unsigned int identifier;
   alsad_connect_addr_t *connect_addr;	// Not sent over network
   alsad_hw_params_t *hw_params;		// Not sent over network
   alsad_sw_params_t *sw_params;		// Not sent over network
};

typedef struct alsad_data_pipe alsad_data_pipe_t;


struct alsad_net_structs{
   alsad_hw_params_t *hw_params;
   alsad_sw_params_t *sw_params;
   alsad_stream_props_t *local_stream;
   alsad_stream_props_t *remote_stream;
   alsad_connect_addr_t *connect_addr;
   alsad_data_pipe_t *local_pipe;
   alsad_data_pipe_t *remote_pipe;
};
 
typedef struct alsad_net_structs alsad_net_structs_t;


void alsad_initialize_hw_params(alsad_hw_params_t *init_hw_params);
void alsad_initialize_sw_params(alsad_sw_params_t *init_sw_params); 
void alsad_initialize_connect_addr(alsad_connect_addr_t *init_connect_addr);
void alsad_initialize_stream_props(alsad_stream_props_t *init_audio_chan);
void alsad_initialize_data_pipe(alsad_data_pipe_t *rcvd_data_pipe);

int alsad_send_hw_params(int sock, alsad_hw_params_t *hd_params);
int alsad_recv_hw_params(int sock, alsad_hw_params_t *hd_params);
int alsad_send_hw_params_w_ctrl(int sock, alsad_hw_params_t *hd_params,
		                           alsad_control_t *send_control);

int alsad_send_stream_props(int sock, alsad_stream_props_t *stream_prop);
int alsad_recv_stream_props(int sock, alsad_stream_props_t *stream_prop);

int alsad_send_conn_addr(int sock, alsad_connect_addr_t *conn_addr);
int alsad_recv_conn_addr(int sock, alsad_connect_addr_t *conn_addr);
int alsad_send_conn_addr_w_ctrl(int sock, alsad_connect_addr_t *conn_addr,
		                           alsad_control_t *send_control);

int alsad_send_sw_params(int sock, alsad_sw_params_t *sw_params);
int alsad_recv_sw_params(int sock, alsad_sw_params_t *sw_params);
int alsad_send_sw_params_w_ctrl(int sock, alsad_sw_params_t *sw_params,
		                           alsad_control_t *send_control);

int alsad_send_audio_hdr(int sock, alsad_audio_hdr_t *audio_hdr, char *audiobuf);
int alsad_recv_audio_hdr(int sock, alsad_audio_hdr_t *audio_hdr);

int alsad_send_control(int sock, alsad_control_t *control);
int alsad_recv_control(int sock, alsad_control_t *control);

/**\todo I should send/recv the hw_params and the sw_params over the network
 * 
 */
int alsad_send_data_pipe(int sock, alsad_data_pipe_t *data_pipe);
int alsad_recv_data_pipe(int sock, alsad_data_pipe_t *data_pipe);
int alsad_send_data_pipe_w_ctrl(int sock, alsad_data_pipe_t *data_pipe,
		                           alsad_control_t *send_control);
		                           
void alsad_free_net_structs(alsad_net_structs_t *free_structs);

int alsad_request_structs(int sock, alsad_net_structs_t *net_structs, 
		                               alsad_control_t *control);
int alsad_reply_structs(int sock, alsad_net_structs_t *net_structs); 


/** @}*/

#endif
