/* quick and dirty spell checker.
	Accepts a list of significant characters; all others are separators.
	Accepts a list of correctly spelled words.
	Uses in-core hashing.  Memory and time intensive.

	Author:  Raphael Finkel 28-Jun-95

	Modifications:
		03-Sep-98: Added options for agrep
			Mods to dictionary file flushed immediately.
			Entered spellings are added to internal dictionary.
		16-Aug-04: Empty line in word list now ignored

*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#ifndef __unix__
#	include <console.h>
#	include <stdlib.h>
#endif

#define true 1
#define false 0
#define Boolean int

#define DEFAULTSFILENAME "defaults" /* contains names of other files */
#define ALPHABETFILENAME "alphabet" /* default name for alphabetfile */
#define INPUTFILENAME "spell.in" /* default name for spell input */
#define OUTPUTFILENAME "spell.out" /* default name for spell output */
#define WORDFILENAME "dictionary" /* default name for dictionary */
#define MAXWORD 100 /* size of longest word expected */
#define BUCKETS 3000 /* number of hash buckets; keep about the size of the 
	dictionary. */

FILE *WordFile; /* dictionary */
char *WordFileName, *AlphabetFileName, *InputFileName, *OutputFileName;
int HashSize;

char *InFileArg = 0, *OutFileArg = 0, *WordFileArg = 0, *AlphFileArg = 0;
int Verbose = 1;

enum {BatchAccept, BatchIgnore, BatchWarn, NoBatch} BatchMode = NoBatch;

/* ------- { utility routines */

#ifndef __unix__
#	define perror(string) fprintf(stdout, "Cannot open %s\n", (string))
#endif

void GetString(char **Which, FILE *File) { 
	char Buffer[MAXWORD]; 
	fgets(Buffer, MAXWORD, File); 
	Buffer[strlen(Buffer)-1] = 0; /* remove \n */ 
	*Which = (char *) malloc(strlen(Buffer)+4); 
	strcpy(*Which, Buffer); 
} /* GetString */

void OpenFile(FILE **FilePtr, char **FileNamePtr, char *FileNameArg,
  char *Mesg, char *How) {
	if (FileNameArg) { /* must be able to open that file, else fail */
		if (strcmp(FileNameArg, "-")) { /* ordinary */
			*FilePtr = fopen(FileNameArg, How);
			if (!*FilePtr) {
				perror(FileNameArg);
				exit(1);
			}
		} else if (strcmp(How, "r") == 0) {
			*FilePtr = stdin;
			/* fprintf(stderr, "Using stdin for input file\n"); */
		} else {
			*FilePtr = stdout;
			/* fprintf(stderr, "Using stdout for output file\n"); */
		}
	} else { /* no arg supplied; interact. */
		char *DefaultFileName = *FileNamePtr;
		for (*FilePtr = 0; !*FilePtr;) { /* until reasonable file specified */
			fprintf(stdout, "%s [default: %s]: ", Mesg, DefaultFileName);
			GetString(FileNamePtr, stdin);
			if (!**FileNamePtr) *FileNamePtr = DefaultFileName;
			*FilePtr = fopen(*FileNamePtr, How);
			if (!*FilePtr) perror(*FileNamePtr);
		}
	} /* interactive */
} /* OpenFile */

/* utility routines } ------- { debugging routines */

Boolean Debug = false;
int WordCount;

void fatal(char *Message){
	fprintf(stdout, "%s\n", Message);
	exit(1);
} /* fatal */

/* debugging routines } ------- { Hash table routines */

typedef struct WordStruct {
		char *Word;
		struct WordStruct *Next;
	} WordNode;

WordNode **HashTable;
WordNode NilNode; /* pseudo-data */

int Hash(char *WordPtr) {
	int Answer = 17;
	char *Ptr = WordPtr;
	for (; *Ptr; Ptr++) {
		Answer = Answer * *Ptr + 1;
	}
	if (Answer < 0) Answer = -Answer;
	if (Debug)
		fprintf(stdout, "%s %d\n", WordPtr, Answer%HashSize);
	return (Answer % HashSize);
} /* Hash */

Boolean Lookup(char *Word){ /* true if found */
	WordNode *Chain = HashTable[Hash(Word)];
	NilNode.Word = Word;
	for (; strcmp(Word, Chain->Word); Chain = Chain->Next);
	return(Chain != &NilNode);
} /* Lookup */


void Insert(char *Word){
	/* assume word is not in the hash table and that it is in temporary space */
	char *NewStore = (char *) malloc(strlen(Word)+1);
	int Index = Hash(Word);
	WordNode *Chain = HashTable[Index];

	strcpy(NewStore, Word);
	HashTable[Index] = (WordNode *) malloc(sizeof(WordNode));
	HashTable[Index]->Word = NewStore;
	HashTable[Index]->Next = Chain;
} /* Insert */

void InitHashTable(){
	int Index;
	char Buffer[MAXWORD];

	WordCount = 0;
	if (WordFileArg) { /* open readonly */
		OpenFile(&WordFile, &WordFileName, WordFileArg, "", "r");
	} else { /* interactive, read/write */
		OpenFile(&WordFile, &WordFileName, WordFileArg,
			"File with your dictionary", "r+");
	}
	for (;;) { /* allocate HashTable until it fits. */
		HashTable = (WordNode **) malloc(sizeof(WordNode) * HashSize);
		if (HashTable != 0) break;
		fprintf(stdout, "Memory seems a bit tight; I will run slower.\n");
		HashSize /= 2;
	} 
	for (Index = 0; Index < HashSize; Index++)
		HashTable[Index] = &NilNode;
	while (fgets(Buffer, MAXWORD, WordFile)) { /* put word in hash table */
		Buffer[strlen(Buffer)-1] = 0; /* remove \n */
		if (strlen(Buffer) == 0) continue; /* ignore empty line */
		if (!Lookup(Buffer)) {
			Insert(Buffer);
			WordCount++;
		}
	}
} /* InitHashTable */

/* Hash table routines } ------- { Delimiter routines */

#define ASCIISIZE 256
#define DELIM 0
#define NONDELIM 1

int CharClass[ASCIISIZE]; /* DELIM or NONDELIM */

void GetDelimiters(){
	int Index;
	int ThisChar;
	FILE *AlphabetFile;

	OpenFile(&AlphabetFile, &AlphabetFileName, AlphFileArg,
		"File with your alphabetic character set", "r");
	for (Index = 0; Index < ASCIISIZE; Index++) CharClass[Index] = DELIM;
	while ((ThisChar = fgetc(AlphabetFile)) > 0) {
		if (ThisChar == '\n') continue;
		if (ThisChar >= ASCIISIZE)
			fatal("Delimiter outside Ascii range.");	
		CharClass[ThisChar] = NONDELIM;
	}
} /* GetDelimiters */

/* delimiter routines } ------- { scanner */

void Suggest(int Distance, char *BadWord) {
	char TmpBuffer[MAXWORD]; 
	fprintf(stdout, "--- suggestions:\n");
	sprintf(TmpBuffer, "agrep -%d \"^%s\\$\" %s" ,
		Distance, BadWord, WordFileName);
	system(TmpBuffer);
	fprintf(stdout, "--- \n");
} /* Suggest */

void Scan(){
	FILE *InputFile;
	FILE *OutputFile;
	char Buffer[MAXWORD], *WordPtr;

	OpenFile(&InputFile, &InputFileName, InFileArg,
		"File you want checked", "r");
	OpenFile(&OutputFile, &OutputFileName, OutFileArg,
		"File to store the corrected text", "w");
	for (;;) { /* each iteration does one word */
		int ThisChar;
		char *Correct;
		WordPtr = Buffer;
		while ((ThisChar = fgetc(InputFile)) >= 0 &&
		  CharClass[ThisChar] == DELIM) {
			fputc(ThisChar, OutputFile);
			if (ThisChar == '\\' || ThisChar == '%') {
				/* TeX insert or comment; skip to end of line */	
				do {
					fputc(ThisChar, OutputFile);
					ThisChar = fgetc(InputFile);
				} while (ThisChar >= 0 && ThisChar != '\n');
				fputc('\n', OutputFile);
			}
		}
		if (ThisChar < 0) break; /* end of file */
		*WordPtr++ = ThisChar;
		ThisChar = fgetc(InputFile);
		for (;
			ThisChar >= 0 && CharClass[ThisChar] == NONDELIM;
			ThisChar = fgetc(InputFile)) *WordPtr++ = ThisChar;
		*WordPtr = 0;
		Correct = Buffer; /* unless we decide to substitute something */
		if (Lookup(Buffer)) {
			if (Debug)
				fprintf(stdout, "spelled correctly: [%s]\n", Buffer);
		} else { /* spelled wrong */
			char AnswerBuffer[10];
			GetAnswer:
			switch (BatchMode) {
				case NoBatch:
#ifndef __unix__
					cgotoxy(1,1, stdout);
					ccleos(stdout);
#endif
					fprintf(stdout, "\nnot spelled correctly: [%s].\n", Buffer);
					fprintf(stdout, "\
G[ood; add to dictionary], I[gnore], E[nter correction], \n\
F[inish but don't add to dictionary], \n\
A[ll from here on good; add to dictionary], \n\
1[ show close matches] 2[ more distant] 3[ quite distant]\n\
\n\
Type one of [GIAFE123]: ");
					fgets(AnswerBuffer, sizeof(AnswerBuffer), stdin);
					break;
				case BatchAccept:
					*AnswerBuffer = 'G'; /* automatically good */
					break;
				case BatchIgnore:
					*AnswerBuffer = 'I'; /* automatically ignore */
					break;
				case BatchWarn:
					fprintf(stderr, "%s ", Buffer);
					*AnswerBuffer = 'I'; /* automatically ignore */
					break;
			} /* switch BatchMode */
			switch (tolower(*AnswerBuffer)) {
				case 'g': case 'a': {
					if (tolower(*AnswerBuffer) == 'a') {
						fprintf(stdout,
							"Treating all words as spelled right.\n");
						BatchMode = BatchAccept;
					}
					Insert(Buffer);
					WordCount++;
					if (BatchMode == NoBatch)
						fprintf(stdout, "Accepting [%s]\n", Buffer);
					fseek(WordFile, 0, SEEK_END); /* maybe someone else has
					added */
					fprintf(WordFile,"%s\n", Buffer);
					fflush(WordFile); /* let others see, too. */
					break;
				}
				case 'f': {
					fprintf(stdout, "Ignoring all spelling to the end.\n");
					BatchMode = BatchIgnore;
					/* fall through to next case */
				}
				case 'i': {
					if (BatchMode == NoBatch)
						fprintf(stdout, "Ignoring [%s]\n", Buffer);
					Insert(Buffer);
					break;
				}
				case 'e': {
					char TmpBuffer[MAXWORD], AnswerBuffer[MAXWORD]; 
					char *NewStore;
					for (;;) { /* get an acceptable spelling */
						fprintf(stdout, "Correct spelling: ");
						fgets(TmpBuffer, MAXWORD, stdin);
						TmpBuffer[strlen(TmpBuffer)-1] = 0; /* remove \n */
						if (Lookup(TmpBuffer)) break; /* looks good */
						fprintf(stdout,
							"[%s] looks wrong.  Are you sure? [Y/N] ",
							TmpBuffer);
						fgets(AnswerBuffer, MAXWORD, stdin);
						if (tolower(*AnswerBuffer) == 'y') {
							Insert(TmpBuffer);
							fseek(WordFile, 0, SEEK_END); /* maybe someone
								else has added */
							fprintf(WordFile,"%s\n", TmpBuffer);
							fflush(WordFile); /* let others see, too. */
							break;
						}
					}
					NewStore = (char *) malloc(strlen(TmpBuffer)+1);
					strcpy(NewStore, TmpBuffer);
					Correct = NewStore;
					break;
				case '1': 
					Suggest(1, Buffer);
					goto GetAnswer;
					break;	
				case '2':
					Suggest(2, Buffer);
					goto GetAnswer;
					break;	
				case '3':
					Suggest(3, Buffer);
					goto GetAnswer;
					break;	
				default:
					goto GetAnswer;
				}
			} /* switch */
		} /* spelled wrong */
		fputs(Correct, OutputFile);
		fputc(ThisChar, OutputFile);
	} /* one word */
	fclose(OutputFile);
	fclose(InputFile);
	fclose(WordFile);
} /* Scan */

/* scanner } ------- { main */

void InitDefaults() {
	FILE *DefaultFile = fopen(DEFAULTSFILENAME, "r");
	if (!DefaultFile) { /* no default file; use built-in defaults */
		AlphabetFileName = ALPHABETFILENAME;
		InputFileName = INPUTFILENAME;
		OutputFileName = OUTPUTFILENAME;
		WordFileName = WORDFILENAME;
		HashSize = BUCKETS;
	} else {
		GetString(&AlphabetFileName, DefaultFile);
		GetString(&InputFileName, DefaultFile);
		GetString(&OutputFileName, DefaultFile);
		GetString(&WordFileName, DefaultFile);
		fscanf(DefaultFile, "%d", &HashSize);
		if (HashSize < BUCKETS) HashSize = BUCKETS;
	}
} /* InitDefaults */

void OutputDefaults() {
	FILE *DefaultFile = fopen(DEFAULTSFILENAME, "w");
	fprintf(DefaultFile, "%s\n%s\n%s\n%s\n\%d\n",
		AlphabetFileName, InputFileName, OutputFileName, WordFileName,
		WordCount);
} /* OutputDefaults */

int main(int argc, char *argv[]) {
	char ThisOpt;

	while ((ThisOpt = getopt(argc, argv, "i:,o:,d:,a:,w,q,g")) != (char) -1) {
		switch (ThisOpt) {
			case 'a':
				AlphFileArg = malloc(strlen(optarg)+1);
				strcpy(AlphFileArg, optarg);
				break;
			case 'i':
				InFileArg = malloc(strlen(optarg)+1);
				strcpy(InFileArg, optarg);
				break;
			case 'o':
				OutFileArg = malloc(strlen(optarg)+1);
				strcpy(OutFileArg, optarg);
				break;
			case 'd':
				WordFileArg = malloc(strlen(optarg)+1);
				strcpy(WordFileArg, optarg);
				break;
			case 'w':
				if (BatchMode == NoBatch)
					BatchMode = BatchWarn;
				else {
					fprintf(stderr, "Can't set both -w and -g\n");
					exit(1);
				}
				break;
			case 'g':
				if (BatchMode == NoBatch)
					BatchMode = BatchAccept;
				else {
					fprintf(stderr, "Can't set both -w and -g\n");
					exit(1);
				}
				break;
			case 'q':
				Verbose = 0;
				break;
			case '?':
				fprintf(stderr,
					"Usage: %s [-i InFile] [-o OutFile] [-d Dict] [-w|g] [-q]\n",
					argv[0]);
				fprintf(stderr, "\t-w means warn of errors, don't fix them.\n");
				fprintf(stderr, "\t-g means accept all words.\n");
				fprintf(stderr, "\t-q turns off final stats.\n");
				exit(1);
		} /* switch ThisOpt */
	} /* while more args */
	InitDefaults();
	InitHashTable();
	GetDelimiters();
	Scan();
	if (Verbose)
		fprintf(stdout, "\nDictionary now has %d words\n", WordCount);
	else
		fprintf(stdout, "\n");
	OutputDefaults();
	return(0);
} /* main */


/* main } ------- */
