" Vim syntax file
" Language:    Yiddish
" Author:      Raphael Finkel <raphael@cs.uky.edu>,
"    based on German by Thomas Khler <jean-luc@picard.franken.de>,
"    based on gerspchk.vim by Dr. Charles E. Campbell, Jr.
"        <Charles.Campbell@gsfc.nasa.gov>
" Last Change: 2001 Apr 2
" Version:     1.00 yiddish
" Download:    German is at http://jeanluc-picard.de/vim/gerspchk/

" Environment Variableg:
"  CVIMSYN: a directory holding the user word database <usr_yidspchk.vim>

" This file defines the following Command Maps:
"  \et : add  word under cursor into database (temporarily - ie. just this file)
"  \es : save word under cursor into user database (permanently)
"  \en : move cursor to the next     spelling error
"  \ep : move cursor to the previous spelling error
"  \ea : look for alternative spellings of word under cursor
"     A list will be shown in an alternatives window.
"     Maps for Alternatives Window Only:
"      <cr> : on alternatives line, will put word under cursor in
"             searchword's stead
"      :q   : will quit the alternate-word window
"  It also introduces this new command:
"  EE <file> : switch to given file in same buffer (so the word list need not
"      be re-read.  Discards current changes without warning.

" Usage:
"  Source this file in.  It does *not* do a "syntax clear", so that means
"  that you can usually just source it in on top of other highlighting.

" Internal Functions: for vim versions 5.4 or later
"  SpchkSave(newword) : saves the word into <$udict>
"  SpchkPrv()         : enables \ep map
"  SpchkNxt()         : enables \en map
"
" History:
" v1.00yiddish: Apr 02 2001: Yiddish
" v1.29german : Feb 24 2001 : German keywords
" v1.29 : Feb  5 2001 : For 6.0v or later: words will no longer flash in Error
"                       as they're being typed.  Thank you Chase Tingley for \%* !
" v1.28 : Jan 29 2001 : shortened SpchkNxt and SpchkPrv (thanks, Benji Fisher!)
" v1.27 : Jan 29 2001 : includes using agrep and \ea for alternative spelling
" v1.26 : Jan 25 2001 : removed a number of two-letter abbreviations & "ain"
" v1.25 : Jan 22 2001 : included 12000 more words from various online dictionaries
"                       I've also culled out a few misspellings that crept in from
"                       the original ispell dictionary that was used.
" v1.24 : Nov  1 2000 : included optional <gerspchk2.vim> sourcing
" v1.22 : Oct  3 2000 : added "transparent" keyword to all GoodWords
" v1.20 : Sep 18 2000 : gerspchk now saves using vim functions, avoiding
"                       any external program entanglements for versions 5.2
"                       or later
" v1.19 : Apr  7 2000 : more contractions and words: couldnt've, etc.
" v1.18 : Mar 28 2000 : included words from spelling bee and other dictionaries
"                       included "didn't" contraction, removed "didn didnt"
" v1.17 : Aug 12 1999 : the epoch
"------------------------------------------------------------------------------

" Constants
let $CVIMSYN=$HOME . "/.vim"
let $udict=$CVIMSYN . "/usr_yidwords.vim"
let $sdict=$CVIMSYN . "/yidwords.vim"
" Maps to facilitate entering new words
"  use  temporarily (\et)
"  save permanently (\es)
map \et :syn case ignore<CR>:exe "syn keyword GoodWord transparent	" . expand("<cword>")<CR>:syn case match<CR>

" \es: saves a new word to a user dictionary ($udict).
if version >= 600
  map \es	:call SpchkSave(expand("<cword>"))<CR>
  fu! SpchkSave(newword)
    silent 1sp
    silent e $udict
    $put='syn keyword GoodWord transparent	'.a:newword
    silent wq
    norm \et
  endfunction

elseif version >= 504
  map \es	:call SpchkSave(expand("<cword>"))<CR>
  fu! SpchkSave(newword)
    1sp
    e $udict
    exe ':$s/$/\rsyn keyword GoodWord transparent	'.a:newword."/"
    wq
    norm \et
  endfunction

else " vim version earlier than 5.2
  map \es	:syn case ignore<CR>:exe "syn keyword GoodWord
  transparent	" . expand("<cword>")<CR>:syn case match<CR>:exe
  '!echo syn keyword GoodWord ' . expand("<cword>") . ' >> ' . expand($udict)<CR>
endif  " vim version test

" Ignore upper/lower case
syn case ignore

" The Dictionary
" ==============
" syn keyword GoodWord transparent longlistofwords
so $sdict
if exists($udict)
  so $udict
endif

" BadWord (misspelling) highlighting done here.  Plus
" comment support -- do spell-checking inside comments.
"                 This can be done only for those syntax
"                 files' comment blocks that contains=@cluster.
"                 The code below adds GoodWord and BadWord to
"                 various clusters.  If your favorite syntax
"                 comments are not included, send a note
"                 to your syntax file's maintainer about it!
if &ft == "c" || &ft == "cpp"
  syn cluster cCommentGroup	add=GoodWord,BadWord
  syn match   BadWord contained	"\<\(\a\|[]\)\+\>"
elseif &ft == "vim"
  syn cluster vimCommentGroup	add=GoodWord,BadWord
  syn match   BadWord contained	"\<\(\a\|[]\)\+\>"
elseif &ft == "tex"
  syn cluster texCommentGroup	add=GoodWord,BadWord
  syn match   BadWord "\<\(\a\|[]\)\+\>"
elseif &ft == "sh" || &ft == "ksh" || &ft == "bash"
  syn cluster shCommentGroup	add=GoodWord,BadWord
  syn match   BadWord contained	"\<\(\a\|[]\)\+\>"
elseif &ft == "csh"
  syn cluster cshCommentGroup	add=GoodWord,BadWord
  syn match   BadWord contained	"\<\(\a\|[]\)\+\>"
elseif &ft == "dcl"
  syn cluster dclCommentGroup	add=GoodWord,BadWord
  syn match   BadWord contained	"\<\(\a\|[]\)\+\>"
elseif &ft == "amiga"
  syn cluster amiCommentGroup	add=GoodWord,BadWord
  syn match   BadWord contained	"\<\(\a\|[]\)\+\>"
else
  " Highlight the BadWords
  syn match   BadWord		"\<\(\a\|[]\)\+\>"
endif

" Slight Single-Letter Abbreviation support
syn match GoodWord		"\<\a\.\(\s\|$\)"

" allows $udict to work better with LaTeX
if &ft == "tex"
  syn match GoodWord	"{[a-zA-Z_|@]\+}"lc=1,me=e-1
  syn match GoodWord	"\[[a-zA-Z_]\+]"lc=1,me=e-1
  syn match texGoodWord	"\\[a-zA-Z_]\+"lc=1,me=e-1	contained
  hi def link texGoodWord texComment
  syn cluster texCommentGroup	add=texGoodWord
endif

" these are words but have special meaning in vim files, so they can't be keywords :o
syn match GoodWord	"\<\(contained\|contains\|display\|fold\|transparent\|skip\)\>"

" ignore web addresses
syn match GoodWord	"\<http://www\.\S\+"

" read user's additional words
 if filereadable(expand("$CVIMSYN/usr_spchk.vim")) > 0
  so $CVIMSYN/usr_spchk.vim
 endif
 if filereadable(expand($udict)) > 0
  so $udict
 endif

syn case match

hi link BadWord	Error

" ignore one-letter words
syn match   GoodWord transparent	"\<.\>"

" introduce EE command to switch files without switching buffer

command! -nargs=1 -bang -complete=file EE call FileSwitch("<args>","x<bang>")

function! FileSwitch(theFile, bang) 
    if &modified && (a:bang == "x")
        echo "File not written; use :EE! " . a:theFile . " to override."
    else
        execute "file" a:theFile
        execute "normal 1GdG"
        execute "0r" a:theFile
        set nomodified
    endif
endfunction

" ---------------------------------------------------------------------
" Functions:  must have vim 5.04 or later
" Support for moving to next/previous spelling error
if version >= 504
  map \en	:call SpchkNxt()<CR>
  map \ep	:call SpchkPrv()<CR>

  " -------------------------------------------------------------------
  " \en : calls this function to search for next spelling error
  function! SpchkNxt()
    let errid   = synIDtrans(hlID("Error"))
    let lastline= line("$")
    let curcol  = 0

    norm w

    " skip words until we find next error
    while synIDtrans(synID(line("."),col("."),1)) != errid
      norm w
      if line(".") == lastline
        let prvcol=curcol
        let curcol=col(".")
        if curcol == prvcol
          break
        endif
      endif
    endwhile

    " cleanup
    unlet curcol
    unlet errid
    unlet lastline
    if exists("prvcol")
      unlet prvcol
    endif
  endfunction

  " -------------------------------------------------------------------
  " \ep : calls this function to search for previous spelling error
  function! SpchkPrv()
    let errid = synIDtrans(hlID("Error"))
    let curcol= 0

    norm b

    " skip words until we find previous error
    while synIDtrans(synID(line("."),col("."),1)) != errid
      norm b
      if line(".") == 1
        let prvcol=curcol
        let curcol=col(".")
        if curcol == prvcol
          break
        endif
      endif
    endwhile

    " cleanup
    unlet curcol
    unlet errid
    if exists("prvcol")
      unlet prvcol
    endif
  endfunction

  if version >= 600
    map \ea :call SpchkAlternate(expand("<cword>"))<CR>

    " Use Chase Tingley patch: prevents Error highlighting
    " of words while one is typing them.  \%* is a new magic
    " atom that matches zero-length if that is where the cursor
    " currently is.
    " syn match GoodWord "\<\a\+\%*\>"
  
    " -----------------------------------------------------------------
    " SpchkAlternate: extract words that are close in spelling
    function! SpchkAlternate(wrd)
    
      if exists("g:esc_alternate")
        " re-use wordlist in bottom window
        exe "norm \<c-w>bG0DAAlternate<".a:wrd.">: \<Esc>"
    
      elseif filereadable($sdict)
        " read in user's wordlist
        let g:esc_alternate= 1
        bo 1new
        setlocal lz
        setlocal winheight=1
        exe "norm GoAlternate<".a:wrd.">: \<Esc>"
      endif
    
      nnoremap <buffer> <CR> <C-W>khea#<Esc>bde<C-W>blbye<C-W>kPlxb
      cnoremap <buffer> q	 :unlet g:esc_alternate<CR>:q!<CR>
      setlocal nomod nowrap ic
    
      " let's add a wee bit of color...
      syn match altLeader	"^Alternate"
      syn match altAngle	"[<>]"
      hi def link altLeader	Statement
      hi def link altAngle	Delimiter
    
      " Build patterns based on permutations of up to first 3 letters
      exe "norm \<c-w>b"
      if strlen(a:wrd) <= 10
        exe "r! agrep -1 -i -w -d' ' ".a:wrd." ".expand("$sdict")
      else
        exe "r! agrep -2 -i -w -d' ' ".a:wrd." ".expand("$sdict")
      endif
      %j
      norm 04w
      setlocal nomod
    endfunction
  endif
else
 map \ea :echo "Not available for version " . version
 map \en :echo "Not available for version " . version
 map \ep :echo "Not available for version " . version
endif

" vim: ts=30
